// ============================================
// TUTORIAL INTERACTIVO - SISTEMA DE VETERINARIA
// Script de funcionalidad interactiva
// ============================================

document.addEventListener('DOMContentLoaded', function() {
  initTabs();
  initNavigation();
  initScrollEffects();
});

// ============================================
// FUNCIONALIDAD DE TABS
// ============================================

function initTabs() {
  const tabButtons = document.querySelectorAll('.tab-btn');
  const tabContents = document.querySelectorAll('.tab-content');

  tabButtons.forEach(button => {
    button.addEventListener('click', function() {
      const tabId = this.getAttribute('data-tab');
      
      // Remover clase active de todos los botones y contenidos
      tabButtons.forEach(btn => btn.classList.remove('active'));
      tabContents.forEach(content => content.classList.remove('active'));
      
      // Agregar clase active al botón y contenido seleccionado
      this.classList.add('active');
      document.getElementById(tabId).classList.add('active');
    });
  });

  // Activar el primer tab por defecto
  if (tabButtons.length > 0) {
    tabButtons[0].click();
  }
}

// ============================================
// NAVEGACIÓN SUAVE
// ============================================

function initNavigation() {
  const navLinks = document.querySelectorAll('a[href^="#"]');
  
  navLinks.forEach(link => {
    link.addEventListener('click', function(e) {
      const href = this.getAttribute('href');
      if (href !== '#') {
        e.preventDefault();
        const target = document.querySelector(href);
        if (target) {
          target.scrollIntoView({ behavior: 'smooth' });
        }
      }
    });
  });
}

// ============================================
// EFECTOS DE SCROLL
// ============================================

function initScrollEffects() {
  const cards = document.querySelectorAll('.card');
  
  const observer = new IntersectionObserver((entries) => {
    entries.forEach(entry => {
      if (entry.isIntersecting) {
        entry.target.style.animation = 'fadeIn 0.6s ease forwards';
      }
    });
  }, {
    threshold: 0.1
  });

  cards.forEach(card => {
    observer.observe(card);
  });
}

// ============================================
// FUNCIONES DE SIMULACIÓN INTERACTIVA
// ============================================

function mostrarSimulacion(tipo) {
  const modal = document.getElementById('simulationModal');
  const content = document.getElementById('simulationContent');
  
  let html = '';
  
  switch(tipo) {
    case 'crear-dueño':
      html = generarFormularioDueño();
      break;
    case 'crear-mascota':
      html = generarFormularioMascota();
      break;
    case 'nueva-visita':
      html = generarFormularioVisita();
      break;
    case 'crear-promocion':
      html = generarFormularioPromocion();
      break;
    default:
      html = '<p>Simulación no disponible</p>';
  }
  
  content.innerHTML = html;
  modal.style.display = 'flex';
}

function cerrarSimulacion() {
  const modal = document.getElementById('simulationModal');
  modal.style.display = 'none';
}

function generarFormularioDueño() {
  return `
    <div class="form-container">
      <h3>Crear Nuevo Dueño</h3>
      <form onsubmit="return mostrarResultado(event, 'dueño')">
        <div class="form-group">
          <label>Nombre Completo:</label>
          <input type="text" placeholder="Ej: Juan García" required>
        </div>
        <div class="form-group">
          <label>Cédula de Identidad:</label>
          <input type="text" placeholder="Ej: 1234567" required>
        </div>
        <div class="form-group">
          <label>Teléfono:</label>
          <input type="tel" placeholder="Ej: +595 981 234567" required>
        </div>
        <div class="form-group">
          <label>Dirección:</label>
          <input type="text" placeholder="Ej: Calle Principal 123">
        </div>
        <div class="form-group">
          <label>Email:</label>
          <input type="email" placeholder="Ej: juan@email.com">
        </div>
        <button type="submit" class="btn btn-primary">Guardar Dueño</button>
      </form>
    </div>
  `;
}

function generarFormularioMascota() {
  return `
    <div class="form-container">
      <h3>Agregar Nueva Mascota</h3>
      <form onsubmit="return mostrarResultado(event, 'mascota')">
        <div class="form-group">
          <label>Nombre de la Mascota:</label>
          <input type="text" placeholder="Ej: Firulais" required>
        </div>
        <div class="form-group">
          <label>Tipo de Mascota:</label>
          <select required>
            <option value="">Seleccionar...</option>
            <option value="perro">Perro</option>
            <option value="gato">Gato</option>
            <option value="conejo">Conejo</option>
            <option value="pajaro">Pájaro</option>
            <option value="otro">Otro</option>
          </select>
        </div>
        <div class="form-group">
          <label>Raza:</label>
          <input type="text" placeholder="Ej: Golden Retriever">
        </div>
        <div class="form-group">
          <label>Fecha de Nacimiento:</label>
          <input type="date" required>
        </div>
        <div class="form-group">
          <label>Peso (kg):</label>
          <input type="number" step="0.1" placeholder="Ej: 25.5">
        </div>
        <button type="submit" class="btn btn-primary">Guardar Mascota</button>
      </form>
    </div>
  `;
}

function generarFormularioVisita() {
  return `
    <div class="form-container">
      <h3>Registrar Nueva Visita</h3>
      <form onsubmit="return mostrarResultado(event, 'visita')">
        <div class="form-group">
          <label>Motivo de la Consulta:</label>
          <input type="text" placeholder="Ej: Vacunación anual" required>
        </div>
        <div class="form-group">
          <label>Diagnóstico:</label>
          <textarea placeholder="Ej: Mascota en buen estado de salud..." required></textarea>
        </div>
        <div class="form-group">
          <label>Tratamiento Aplicado:</label>
          <select required>
            <option value="">Seleccionar...</option>
            <option value="consulta">Consulta General - Gs. 150.000</option>
            <option value="vacuna">Vacunación - Gs. 200.000</option>
            <option value="limpieza">Limpieza Dental - Gs. 300.000</option>
            <option value="cirugia">Cirugía - Gs. 500.000</option>
          </select>
        </div>
        <div class="form-group">
          <label>Medicamento Prescrito:</label>
          <select>
            <option value="">Ninguno</option>
            <option value="amoxicilina">Amoxicilina 500mg - Gs. 25.000</option>
            <option value="ibuprofeno">Ibuprofeno 200mg - Gs. 15.000</option>
            <option value="vitaminas">Vitaminas - Gs. 35.000</option>
          </select>
        </div>
        <button type="submit" class="btn btn-primary">Guardar Visita</button>
      </form>
    </div>
  `;
}

function generarFormularioPromocion() {
  return `
    <div class="form-container">
      <h3>Crear Nueva Promoción</h3>
      <form onsubmit="return mostrarResultado(event, 'promocion')">
        <div class="form-group">
          <label>Nombre de la Promoción:</label>
          <input type="text" placeholder="Ej: Descuento en Vacunas" required>
        </div>
        <div class="form-group">
          <label>Descripción:</label>
          <textarea placeholder="Ej: Descuento especial en vacunas..." required></textarea>
        </div>
        <div class="form-group">
          <label>Tipo de Descuento:</label>
          <select required>
            <option value="">Seleccionar...</option>
            <option value="porcentaje">Porcentaje (%)</option>
            <option value="monto">Monto Fijo (Gs.)</option>
          </select>
        </div>
        <div class="form-group">
          <label>Valor del Descuento:</label>
          <input type="number" step="0.01" placeholder="Ej: 15 o 50000" required>
        </div>
        <div class="form-group">
          <label>Fecha de Inicio:</label>
          <input type="date" required>
        </div>
        <div class="form-group">
          <label>Fecha de Fin:</label>
          <input type="date" required>
        </div>
        <button type="submit" class="btn btn-primary">Crear Promoción</button>
      </form>
    </div>
  `;
}

function mostrarResultado(event, tipo) {
  event.preventDefault();
  
  const formData = new FormData(event.target);
  const datos = Object.fromEntries(formData);
  
  let mensaje = '';
  switch(tipo) {
    case 'dueño':
      mensaje = `✅ Dueño "${datos.nombre}" registrado exitosamente con cédula ${datos.cedula}`;
      break;
    case 'mascota':
      mensaje = `✅ Mascota "${datos.nombre}" agregada correctamente`;
      break;
    case 'visita':
      mensaje = `✅ Visita registrada: ${datos.motivo}`;
      break;
    case 'promocion':
      mensaje = `✅ Promoción creada exitosamente`;
      break;
  }
  
  const content = document.getElementById('simulationContent');
  content.innerHTML = `
    <div class="alert alert-success">
      <h3>${mensaje}</h3>
      <p>Este es un ejemplo de simulación. Los datos no se guardan realmente.</p>
      <button class="btn btn-primary" onclick="cerrarSimulacion()">Cerrar</button>
    </div>
  `;
}

// ============================================
// CERRAR MODAL AL HACER CLIC FUERA
// ============================================

window.addEventListener('click', function(event) {
  const modal = document.getElementById('simulationModal');
  if (modal && event.target === modal) {
    cerrarSimulacion();
  }
});
