<?php
require_once 'config.php';

if (!isLoggedIn() || !hasRole('Operativo')) {
    redirect('dashboard.php');
}

$db = getDB();
$mensaje = '';
$error = '';

// Obtener información del usuario actual
$stmt = $db->prepare("SELECT u.*, r.nombre as rol_nombre FROM usuarios u 
                      INNER JOIN roles r ON u.rol_id = r.id 
                      WHERE u.id = ?");
$stmt->execute([$_SESSION['user_id']]);
$usuario = $stmt->fetch();

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $categoria_id = (int)$_POST['categoria_id'];
    $tipo = cleanInput($_POST['tipo']);
    $monto = (float)str_replace(['.', ','], ['', '.'], $_POST['monto']);
    $descripcion = cleanInput($_POST['descripcion']);
    $fecha_movimiento = cleanInput($_POST['fecha_movimiento']);
    
    // Validar que sea un egreso (los operativos solo registran egresos)
    if ($tipo !== 'Egreso') {
        $error = 'Los operativos solo pueden registrar egresos';
    } elseif ($monto <= 0) {
        $error = 'El monto debe ser mayor a cero';
    } elseif ($monto > $usuario['saldo_disponible']) {
        $error = 'El monto supera el saldo disponible';
    } else {
        // Procesar imagen del comprobante
        $comprobante = null;
        if (isset($_FILES['comprobante']) && $_FILES['comprobante']['error'] === UPLOAD_ERR_OK) {
            $allowed = ['jpg', 'jpeg', 'png', 'gif'];
            $filename = $_FILES['comprobante']['name'];
            $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
            
            if (in_array($ext, $allowed)) {
                $new_filename = uniqid() . '_' . time() . '.' . $ext;
                $upload_path = 'uploads/' . $new_filename;
                
                if (move_uploaded_file($_FILES['comprobante']['tmp_name'], $upload_path)) {
                    $comprobante = $new_filename;
                }
            }
        }
        
        try {
            $db->beginTransaction();
            
            // Calcular saldos
            $saldo_anterior = $usuario['saldo_disponible'];
            $saldo_posterior = $saldo_anterior - $monto;
            
            // Insertar movimiento
            $stmt = $db->prepare("INSERT INTO movimientos (usuario_id, categoria_id, tipo, monto, descripcion, 
                                  comprobante, fecha_movimiento, saldo_anterior, saldo_posterior, created_by) 
                                  VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
            $stmt->execute([$_SESSION['user_id'], $categoria_id, $tipo, $monto, $descripcion, 
                           $comprobante, $fecha_movimiento, $saldo_anterior, $saldo_posterior, $_SESSION['user_id']]);
            
            // Actualizar saldo del usuario
            $stmt = $db->prepare("UPDATE usuarios SET saldo_disponible = ? WHERE id = ?");
            $stmt->execute([$saldo_posterior, $_SESSION['user_id']]);
            
            // Actualizar saldo del supervisor
            if ($usuario['supervisor_id']) {
                $stmt = $db->prepare("UPDATE usuarios SET saldo_disponible = saldo_disponible - ? 
                                      WHERE id = ?");
                $stmt->execute([$monto, $usuario['supervisor_id']]);
            }
            
            // Log
            $stmt = $db->prepare("INSERT INTO logs (usuario_id, accion, detalles, ip_address) VALUES (?, ?, ?, ?)");
            $stmt->execute([$_SESSION['user_id'], 'Registrar movimiento', "Monto: $monto", $_SERVER['REMOTE_ADDR']]);
            
            $db->commit();
            
            $mensaje = 'Movimiento registrado exitosamente';
            
            // Actualizar datos del usuario
            $stmt = $db->prepare("SELECT saldo_disponible FROM usuarios WHERE id = ?");
            $stmt->execute([$_SESSION['user_id']]);
            $usuario = array_merge($usuario, $stmt->fetch());
            
        } catch (Exception $e) {
            $db->rollBack();
            $error = 'Error al registrar el movimiento: ' . $e->getMessage();
        }
    }
}

// Obtener categorías de egreso
$categorias = $db->query("SELECT * FROM categorias WHERE tipo = 'Egreso' AND activo = 1 ORDER BY nombre")->fetchAll();

$page_title = 'Nuevo Movimiento';
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?> - <?php echo SITE_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <?php include 'includes/navbar.php'; ?>
    
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">
                        <i class="bi bi-plus-circle"></i> Nuevo Movimiento
                    </h1>
                </div>
                
                <?php if ($mensaje): ?>
                    <div class="alert alert-success alert-dismissible fade show" role="alert">
                        <i class="bi bi-check-circle"></i> <?php echo $mensaje; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>
                
                <?php if ($error): ?>
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <i class="bi bi-exclamation-triangle"></i> <?php echo $error; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>
                
                <!-- Saldo disponible -->
                <div class="row mb-4">
                    <div class="col-md-12">
                        <div class="card stat-card stat-success">
                            <div class="card-body">
                                <div class="d-flex justify-content-between">
                                    <div>
                                        <h6 class="text-muted mb-2">Saldo Disponible</h6>
                                        <h2 class="mb-0"><?php echo formatCurrency($usuario['saldo_disponible']); ?></h2>
                                    </div>
                                    <div class="stat-icon">
                                        <i class="bi bi-wallet2"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0"><i class="bi bi-receipt"></i> Registrar Egreso</h5>
                    </div>
                    <div class="card-body">
                        <form method="POST" enctype="multipart/form-data">
                            <input type="hidden" name="tipo" value="Egreso">
                            
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">Categoría *</label>
                                    <select class="form-select" name="categoria_id" required>
                                        <option value="">Seleccione una categoría</option>
                                        <?php foreach ($categorias as $cat): ?>
                                            <option value="<?php echo $cat['id']; ?>">
                                                <?php echo htmlspecialchars($cat['nombre']); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">Fecha *</label>
                                    <input type="date" class="form-control" name="fecha_movimiento" 
                                           value="<?php echo date('Y-m-d'); ?>" required>
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label">Monto (Gs.) *</label>
                                <input type="text" class="form-control" name="monto" id="monto" 
                                       placeholder="0" required>
                                <small class="text-muted">Ingrese el monto sin puntos ni comas</small>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label">Descripción *</label>
                                <textarea class="form-control" name="descripcion" rows="3" 
                                          placeholder="Describa el gasto realizado" required></textarea>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label">Comprobante (Foto)</label>
                                <input type="file" class="form-control" name="comprobante" 
                                       accept="image/*" capture="camera" id="comprobante">
                                <small class="text-muted">Puede tomar una foto o seleccionar desde la galería</small>
                                <div id="preview"></div>
                            </div>
                            
                            <div class="d-grid gap-2">
                                <button type="submit" class="btn btn-primary btn-lg">
                                    <i class="bi bi-save"></i> Registrar Movimiento
                                </button>
                                <a href="mis_movimientos.php" class="btn btn-secondary">
                                    <i class="bi bi-arrow-left"></i> Volver
                                </a>
                            </div>
                        </form>
                    </div>
                </div>
            </main>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Formatear monto con separadores de miles
        document.getElementById('monto').addEventListener('input', function(e) {
            let value = e.target.value.replace(/\D/g, '');
            e.target.value = value.replace(/\B(?=(\d{3})+(?!\d))/g, '.');
        });
        
        // Preview de imagen
        document.getElementById('comprobante').addEventListener('change', function(e) {
            const preview = document.getElementById('preview');
            preview.innerHTML = '';
            
            if (e.target.files && e.target.files[0]) {
                const reader = new FileReader();
                reader.onload = function(e) {
                    const img = document.createElement('img');
                    img.src = e.target.result;
                    img.className = 'image-preview';
                    preview.appendChild(img);
                };
                reader.readAsDataURL(e.target.files[0]);
            }
        });
    </script>
</body>
</html>

