<?php
require_once 'config.php';

if (!isLoggedIn()) {
    redirect('index.php');
}

$db = getDB();

// Obtener información del usuario actual
$stmt = $db->prepare("SELECT u.*, r.nombre as rol_nombre FROM usuarios u 
                      INNER JOIN roles r ON u.rol_id = r.id 
                      WHERE u.id = ?");
$stmt->execute([$_SESSION['user_id']]);
$usuario = $stmt->fetch();

// Calcular estadísticas según el rol
$stats = [];

if ($usuario['rol_nombre'] === 'Administrador') {
    // 1. Total del Proyecto - Fondo total asignado al proyecto (fijo, solo cambia cuando admin agrega más fondos)
    $stmt = $db->query("SELECT valor FROM configuracion WHERE clave = 'fondo_total_proyecto'");
    $fondo_total = $stmt->fetchColumn();
    $stats['total_proyecto'] = $fondo_total ? (float)$fondo_total : 0;
    
    // 2. Saldo Disponible - Se resta cada vez que se asigna fondos a supervisores
    // Es el fondo total menos lo que se ha asignado a supervisores
    $stmt = $db->query("SELECT SUM(fondo_asignado) as total_asignado_supervisores 
                        FROM usuarios WHERE rol_id = 2 AND activo = 1");
    $asignado_supervisores = $stmt->fetch();
    $total_asignado = $asignado_supervisores['total_asignado_supervisores'] ?? 0;
    $stats['saldo_disponible'] = $stats['total_proyecto'] - $total_asignado;
    
    // 3. Total Gastado - Suma de todos los egresos de los operativos
    $stmt = $db->query("SELECT COALESCE(SUM(m.monto), 0) as total_gastado 
                        FROM movimientos m 
                        INNER JOIN usuarios u ON m.usuario_id = u.id 
                        WHERE u.rol_id = 3 AND m.tipo = 'Egreso'");
    $stats['total_gastado'] = $stmt->fetchColumn();
    
    // Contar usuarios por rol
    $stmt = $db->query("SELECT r.nombre, COUNT(*) as total FROM usuarios u 
                        INNER JOIN roles r ON u.rol_id = r.id 
                        WHERE u.activo = 1 GROUP BY r.nombre");
    $stats['usuarios_por_rol'] = $stmt->fetchAll();
    
    // Movimientos recientes
    $stmt = $db->query("SELECT m.*, u.nombre_completo, c.nombre as categoria 
                        FROM movimientos m 
                        INNER JOIN usuarios u ON m.usuario_id = u.id 
                        INNER JOIN categorias c ON m.categoria_id = c.id 
                        ORDER BY m.created_at DESC LIMIT 10");
    $stats['movimientos_recientes'] = $stmt->fetchAll();
    
} elseif ($usuario['rol_nombre'] === 'Supervisor') {
    // 4. Fondo Asignado - Fijo, lo que le asignó el administrador
    $stats['fondo_asignado'] = $usuario['fondo_asignado'];
    
    // 5. Saldo Disponible - Se resta cada vez que asigna a un operativo
    // Es el fondo asignado menos lo que ha asignado a sus operativos
    $stmt = $db->prepare("SELECT COALESCE(SUM(fondo_asignado), 0) as total_asignado_operativos 
                          FROM usuarios WHERE supervisor_id = ? AND activo = 1");
    $stmt->execute([$usuario['id']]);
    $asignado_operativos = $stmt->fetchColumn();
    $stats['saldo_disponible'] = $stats['fondo_asignado'] - $asignado_operativos;
    
    // 6. Total Gastado - Lo que van gastando cada uno de sus operativos
    $stmt = $db->prepare("SELECT COALESCE(SUM(m.monto), 0) as total_gastado 
                          FROM movimientos m 
                          INNER JOIN usuarios u ON m.usuario_id = u.id 
                          WHERE u.supervisor_id = ? AND m.tipo = 'Egreso'");
    $stmt->execute([$usuario['id']]);
    $stats['total_gastado'] = $stmt->fetchColumn();
    
    // Operativos bajo su supervisión
    $stmt = $db->prepare("SELECT COUNT(*) as total FROM usuarios 
                          WHERE supervisor_id = ? AND activo = 1");
    $stmt->execute([$usuario['id']]);
    $stats['total_operativos'] = $stmt->fetchColumn();
    
    // Total asignado a operativos
    $stats['fondos_operativos'] = $asignado_operativos;
    
    // Movimientos recientes de sus operativos
    $stmt = $db->prepare("SELECT m.*, u.nombre_completo, c.nombre as categoria 
                          FROM movimientos m 
                          INNER JOIN usuarios u ON m.usuario_id = u.id 
                          INNER JOIN categorias c ON m.categoria_id = c.id 
                          WHERE u.supervisor_id = ? OR m.usuario_id = ?
                          ORDER BY m.created_at DESC LIMIT 10");
    $stmt->execute([$usuario['id'], $usuario['id']]);
    $stats['movimientos_recientes'] = $stmt->fetchAll();
    
} else { // Operativo
    // Fondos del operativo
    $stats['fondo_asignado'] = $usuario['fondo_asignado'];
    $stats['saldo_disponible'] = $usuario['saldo_disponible'];
    $stats['total_gastado'] = $stats['fondo_asignado'] - $stats['saldo_disponible'];
    
    // Movimientos del operativo
    $stmt = $db->prepare("SELECT COUNT(*) as total FROM movimientos WHERE usuario_id = ?");
    $stmt->execute([$usuario['id']]);
    $stats['total_movimientos'] = $stmt->fetchColumn();
    
    // Movimientos recientes
    $stmt = $db->prepare("SELECT m.*, c.nombre as categoria 
                          FROM movimientos m 
                          INNER JOIN categorias c ON m.categoria_id = c.id 
                          WHERE m.usuario_id = ? 
                          ORDER BY m.created_at DESC LIMIT 10");
    $stmt->execute([$usuario['id']]);
    $stats['movimientos_recientes'] = $stmt->fetchAll();
}

$page_title = 'Dashboard';
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?> - <?php echo SITE_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <?php include 'includes/navbar.php'; ?>
    
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">
                        <i class="bi bi-speedometer2"></i> Dashboard
                    </h1>
                    <div class="user-info">
                        <span class="badge bg-success"><?php echo htmlspecialchars($usuario['rol_nombre']); ?></span>
                        <strong><?php echo htmlspecialchars($usuario['nombre_completo']); ?></strong>
                    </div>
                </div>
                
                <!-- Tarjetas de estadísticas -->
                <div class="row mb-4">
                    <?php if ($usuario['rol_nombre'] === 'Administrador'): ?>
                        <div class="col-md-4 mb-3">
                            <div class="card stat-card stat-primary">
                                <div class="card-body">
                                    <div class="d-flex justify-content-between">
                                        <div>
                                            <h6 class="text-muted mb-2">Total del Proyecto</h6>
                                            <h3 class="mb-0"><?php echo formatCurrency($stats['total_proyecto']); ?></h3>
                                            <small class="text-muted">Fondo inicial asignado</small>
                                        </div>
                                        <div class="stat-icon">
                                            <i class="bi bi-cash-stack"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-md-4 mb-3">
                            <div class="card stat-card stat-success">
                                <div class="card-body">
                                    <div class="d-flex justify-content-between">
                                        <div>
                                            <h6 class="text-muted mb-2">Saldo Disponible</h6>
                                            <h3 class="mb-0"><?php echo formatCurrency($stats['saldo_disponible']); ?></h3>
                                            <small class="text-muted">Para asignar a supervisores</small>
                                        </div>
                                        <div class="stat-icon">
                                            <i class="bi bi-wallet2"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-md-4 mb-3">
                            <div class="card stat-card stat-danger">
                                <div class="card-body">
                                    <div class="d-flex justify-content-between">
                                        <div>
                                            <h6 class="text-muted mb-2">Total Gastado</h6>
                                            <h3 class="mb-0"><?php echo formatCurrency($stats['total_gastado']); ?></h3>
                                            <small class="text-muted">Gastos de operativos</small>
                                        </div>
                                        <div class="stat-icon">
                                            <i class="bi bi-graph-down-arrow"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    <?php else: ?>
                        <div class="col-md-4 mb-3">
                            <div class="card stat-card stat-primary">
                                <div class="card-body">
                                    <div class="d-flex justify-content-between">
                                        <div>
                                            <h6 class="text-muted mb-2">Fondo Asignado</h6>
                                            <h3 class="mb-0"><?php echo formatCurrency($stats['fondo_asignado']); ?></h3>
                                            <small class="text-muted">Total recibido</small>
                                        </div>
                                        <div class="stat-icon">
                                            <i class="bi bi-cash-stack"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-md-4 mb-3">
                            <div class="card stat-card stat-success">
                                <div class="card-body">
                                    <div class="d-flex justify-content-between">
                                        <div>
                                            <h6 class="text-muted mb-2">Saldo Disponible</h6>
                                            <h3 class="mb-0"><?php echo formatCurrency($stats['saldo_disponible']); ?></h3>
                                            <small class="text-muted">
                                                <?php echo $usuario['rol_nombre'] === 'Supervisor' ? 'Para asignar' : 'Para gastar'; ?>
                                            </small>
                                        </div>
                                        <div class="stat-icon">
                                            <i class="bi bi-wallet2"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-md-4 mb-3">
                            <div class="card stat-card stat-danger">
                                <div class="card-body">
                                    <div class="d-flex justify-content-between">
                                        <div>
                                            <h6 class="text-muted mb-2">Total Gastado</h6>
                                            <h3 class="mb-0"><?php echo formatCurrency($stats['total_gastado']); ?></h3>
                                            <small class="text-muted">
                                                <?php echo $usuario['rol_nombre'] === 'Supervisor' ? 'Gastos de operativos' : 'Mis gastos'; ?>
                                            </small>
                                        </div>
                                        <div class="stat-icon">
                                            <i class="bi bi-graph-down-arrow"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    <?php endif; ?>
                </div>
                
                <!-- Movimientos recientes -->
                <div class="card mb-4">
                    <div class="card-header">
                        <h5 class="mb-0"><i class="bi bi-clock-history"></i> Movimientos Recientes</h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Fecha</th>
                                        <?php if ($usuario['rol_nombre'] !== 'Operativo'): ?>
                                            <th>Usuario</th>
                                        <?php endif; ?>
                                        <th>Categoría</th>
                                        <th>Tipo</th>
                                        <th>Monto</th>
                                        <th>Descripción</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if (!empty($stats['movimientos_recientes'])): ?>
                                        <?php foreach ($stats['movimientos_recientes'] as $mov): ?>
                                            <tr>
                                                <td><?php echo date('d/m/Y', strtotime($mov['fecha_movimiento'])); ?></td>
                                                <?php if ($usuario['rol_nombre'] !== 'Operativo'): ?>
                                                    <td><?php echo htmlspecialchars($mov['nombre_completo']); ?></td>
                                                <?php endif; ?>
                                                <td><?php echo htmlspecialchars($mov['categoria']); ?></td>
                                                <td>
                                                    <?php if ($mov['tipo'] === 'Ingreso'): ?>
                                                        <span class="badge bg-success">Ingreso</span>
                                                    <?php else: ?>
                                                        <span class="badge bg-danger">Egreso</span>
                                                    <?php endif; ?>
                                                </td>
                                                <td><?php echo formatCurrency($mov['monto']); ?></td>
                                                <td><?php echo htmlspecialchars(substr($mov['descripcion'], 0, 50)); ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    <?php else: ?>
                                        <tr>
                                            <td colspan="6" class="text-center text-muted">No hay movimientos registrados</td>
                                        </tr>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

