<?php
/**
 * Página de inicio - Autenticación de clientes
 * Sistema de Supermercados y Tiendas V3.0
 */

require_once 'config/database.php';
require_once 'config/functions.php';

iniciarSesion();

// Si ya está autenticado, redirigir al dashboard
if (estaAutenticadoCliente()) {
    redirigir('cliente/dashboard.php');
}

$error = '';
$mensaje = '';

// Procesar formulario de autenticación
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['cedula_ruc'])) {
    $cedula = limpiarCedulaRuc($_POST['cedula_ruc']);
    
    if (validarCedulaRuc($cedula)) {
        try {
            $db = getDB();
            $stmt = $db->prepare("SELECT * FROM clientes WHERE cedula_ruc = ?");
            $stmt->execute([$cedula]);
            $cliente = $stmt->fetch();
            
            if ($cliente) {
                // Cliente encontrado - iniciar sesión
                $_SESSION['cliente_id'] = $cliente['id'];
                $_SESSION['cliente_nombre'] = $cliente['nombre'];
                $_SESSION['cliente_apellido'] = $cliente['apellido'];
                $_SESSION['cliente_cedula'] = $cliente['cedula_ruc'];
                
                redirigir('cliente/dashboard.php');
            } else {
                // Cliente no encontrado - redirigir a registro
                $_SESSION['registro_cedula'] = $cedula;
                redirigir('cliente/registro.php');
            }
        } catch (PDOException $e) {
            $error = 'Error al verificar la cédula. Intente nuevamente.';
        }
    } else {
        $error = 'Formato de cédula/RUC inválido. Ejemplos válidos: 1234567, 123456-7 (cédula) o 12345678-9, 123456789 (RUC)';
    }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Sistema de Tiendas V3.0 - Bienvenido</title>
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body class="login-page">
    <div class="login-container">
        <div class="logo-container">
            <img src="assets/images/logo.png" alt="Logo Tienda" class="logo" onerror="this.style.display='none'">
            <h1>Sistema de Tiendas V3.0</h1>
        </div>
        
        <div class="login-card">
            <h2>Bienvenido</h2>
            <p class="subtitle">Ingrese su cédula o RUC para continuar</p>
            
            <?php if ($error): ?>
                <div class="alert alert-error">
                    <?php echo htmlspecialchars($error); ?>
                </div>
            <?php endif; ?>
            
            <?php if ($mensaje): ?>
                <div class="alert alert-success">
                    <?php echo htmlspecialchars($mensaje); ?>
                </div>
            <?php endif; ?>
            
            <form method="POST" action="" class="login-form">
                <div class="form-group">
                    <label for="cedula_ruc">Cédula o RUC</label>
                    <input 
                        type="text" 
                        id="cedula_ruc" 
                        name="cedula_ruc" 
                        class="form-control form-control-lg" 
                        placeholder="Ej: 1234567 (cédula) o 80012345-6 (RUC)"
                        required
                        autofocus
                        pattern="[0-9\-]+"
                        title="Solo números y guiones"
                    >
                    <small class="form-text">Ingrese su cédula (6-7 dígitos) o RUC (8-9 dígitos), con o sin guión</small>
                </div>
                
                <button type="submit" class="btn btn-primary btn-lg btn-block">
                    Ingresar
                </button>
            </form>
            
            <div class="login-footer">
                <p>Si no está registrado, se le solicitará completar un breve formulario</p>
            </div>
        </div>
        
        <div class="admin-link">
            <a href="admin/login.php">Acceso Administración</a>
        </div>
    </div>
    
    <script src="assets/js/main.js"></script>
    <script>
        // Limpiar espacios automáticamente al escribir cédula
        document.getElementById('cedula_ruc').addEventListener('input', function(e) {
            // Eliminar todos los espacios
            this.value = this.value.replace(/\s/g, '');
        });
        
        // Limpiar espacios antes de enviar el formulario
        document.querySelector('.login-form').addEventListener('submit', function(e) {
            const cedulaInput = document.getElementById('cedula_ruc');
            cedulaInput.value = cedulaInput.value.trim().replace(/\s/g, '');
        });
    </script>
</body>
</html>
