<?php
/**
 * Funciones auxiliares del sistema
 * Sistema de Supermercados y Tiendas V3.0
 */

// Iniciar sesión si no está iniciada
function iniciarSesion() {
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }
}

// Formatear número como moneda en guaraníes
function formatearGuaranies($monto) {
    return '₲ ' . number_format($monto, 0, ',', '.');
}

// Limpiar y validar cédula/RUC
function limpiarCedulaRuc($cedula) {
    // Eliminar espacios en blanco al inicio y final
    $cedula = trim($cedula);
    // Eliminar todos los espacios en blanco
    $cedula = preg_replace('/\s+/', '', $cedula);
    // Eliminar todo excepto números y guiones
    $cedula = preg_replace('/[^0-9\-]/', '', $cedula);
    // Eliminar guiones duplicados o múltiples
    $cedula = preg_replace('/-+/', '-', $cedula);
    // Eliminar guiones al inicio o final
    $cedula = trim($cedula, '-');
    
    return $cedula;
}

// Validar formato de cédula/RUC paraguayo
function validarCedulaRuc($cedula) {
    // Limpiar la cédula/RUC
    $limpia = limpiarCedulaRuc($cedula);
    
    // Extraer solo los números (sin guión)
    $soloNumeros = preg_replace('/[^0-9]/', '', $limpia);
    
    // Validar longitud: cédula 6-7 dígitos, RUC 8-9 dígitos
    $longitud = strlen($soloNumeros);
    
    // Debe tener entre 6 y 10 dígitos (incluyendo dígito verificador)
    if ($longitud < 6 || $longitud > 10) {
        return false;
    }
    
    // Debe contener solo números (después de limpiar)
    if (!ctype_digit($soloNumeros)) {
        return false;
    }
    
    // Validar formato con guión (opcional)
    // Formatos válidos: 1234567, 123456-7, 12345678-9, 123456789
    // Solo se permite un guión antes del último dígito
    $patron = '/^[0-9]{5,9}(-[0-9]{1})?$/';
    
    return preg_match($patron, $limpia) === 1;
}

// Formatear cédula para mostrar (agregar guión antes del último dígito)
function formatearCedula($cedula) {
    $cedula = limpiarCedulaRuc($cedula);
    if (strlen($cedula) > 1) {
        return substr($cedula, 0, -1) . '-' . substr($cedula, -1);
    }
    return $cedula;
}

// Sanitizar entrada de texto
function sanitizar($texto) {
    return htmlspecialchars(strip_tags(trim($texto)), ENT_QUOTES, 'UTF-8');
}

// Verificar si el usuario está autenticado (cliente)
function estaAutenticadoCliente() {
    iniciarSesion();
    return isset($_SESSION['cliente_id']) && !empty($_SESSION['cliente_id']);
}

// Verificar si el usuario está autenticado (admin)
function estaAutenticadoAdmin() {
    iniciarSesion();
    return isset($_SESSION['admin_id']) && !empty($_SESSION['admin_id']);
}

// Obtener datos del cliente actual
function obtenerClienteActual() {
    iniciarSesion();
    if (estaAutenticadoCliente()) {
        return [
            'id' => $_SESSION['cliente_id'],
            'nombre' => $_SESSION['cliente_nombre'] ?? '',
            'apellido' => $_SESSION['cliente_apellido'] ?? '',
            'cedula_ruc' => $_SESSION['cliente_cedula'] ?? ''
        ];
    }
    return null;
}

// Obtener datos del admin actual
function obtenerAdminActual() {
    iniciarSesion();
    if (estaAutenticadoAdmin()) {
        return [
            'id' => $_SESSION['admin_id'],
            'username' => $_SESSION['admin_username'] ?? '',
            'nombre' => $_SESSION['admin_nombre'] ?? '',
            'rol' => $_SESSION['admin_rol'] ?? ''
        ];
    }
    return null;
}

// Verificar permisos según rol
function tienePermiso($accion) {
    $admin = obtenerAdminActual();
    if (!$admin) return false;
    
    $rol = $admin['rol'];
    
    $permisos = [
        'administrador' => ['crear', 'modificar', 'eliminar', 'ver', 'gestionar_usuarios', 'gestionar_clientes', 'gestionar_promociones', 'enviar_mensajes', 'ver_reportes', 'ver_compras'],
        'comercial' => ['crear', 'modificar', 'eliminar', 'ver', 'gestionar_promociones', 'enviar_mensajes', 'ver_compras'],
        'colaborador' => ['crear', 'ver', 'ver_compras']
    ];
    
    return isset($permisos[$rol]) && in_array($accion, $permisos[$rol]);
}

// Redirigir a una página
function redirigir($url) {
    header("Location: $url");
    exit();
}

// Generar respuesta JSON
function responderJSON($datos, $codigo = 200) {
    http_response_code($codigo);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($datos, JSON_UNESCAPED_UNICODE);
    exit();
}

// Registrar actividad en log (opcional)
function registrarLog($mensaje, $tipo = 'info') {
    $fecha = date('Y-m-d H:i:s');
    $log = "[$fecha] [$tipo] $mensaje" . PHP_EOL;
    file_put_contents(__DIR__ . '/../logs/sistema.log', $log, FILE_APPEND);
}

// Generar token CSRF
function generarTokenCSRF() {
    iniciarSesion();
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

// Verificar token CSRF
function verificarTokenCSRF($token) {
    iniciarSesion();
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

// Subir imagen
function subirImagen($archivo, $directorio = 'uploads/productos/') {
    $directorioCompleto = __DIR__ . '/../' . $directorio;
    
    // Crear directorio si no existe
    if (!file_exists($directorioCompleto)) {
        mkdir($directorioCompleto, 0755, true);
    }
    
    // Validar que sea una imagen
    $tiposPermitidos = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
    if (!in_array($archivo['type'], $tiposPermitidos)) {
        return ['error' => 'Tipo de archivo no permitido'];
    }
    
    // Validar tamaño (máximo 5MB)
    if ($archivo['size'] > 5 * 1024 * 1024) {
        return ['error' => 'El archivo es demasiado grande'];
    }
    
    // Generar nombre único
    $extension = pathinfo($archivo['name'], PATHINFO_EXTENSION);
    $nombreArchivo = uniqid() . '_' . time() . '.' . $extension;
    $rutaCompleta = $directorioCompleto . $nombreArchivo;
    
    // Mover archivo
    if (move_uploaded_file($archivo['tmp_name'], $rutaCompleta)) {
        return ['success' => true, 'ruta' => $directorio . $nombreArchivo];
    }
    
    return ['error' => 'Error al subir el archivo'];
}

// Obtener fecha y hora actual en formato Paraguay
function obtenerFechaHoraActual() {
    date_default_timezone_set('America/Asuncion');
    return date('Y-m-d H:i:s');
}

// Calcular tiempo transcurrido
function tiempoTranscurrido($fecha) {
    $ahora = new DateTime();
    $anterior = new DateTime($fecha);
    $diferencia = $ahora->diff($anterior);
    
    if ($diferencia->d > 0) {
        return $diferencia->d . ' día' . ($diferencia->d > 1 ? 's' : '');
    } elseif ($diferencia->h > 0) {
        return $diferencia->h . ' hora' . ($diferencia->h > 1 ? 's' : '');
    } elseif ($diferencia->i > 0) {
        return $diferencia->i . ' minuto' . ($diferencia->i > 1 ? 's' : '');
    } else {
        return 'Hace un momento';
    }
}
?>
