<?php
/**
 * Página de registro de nuevos clientes
 * Sistema de Supermercados y Tiendas V3.0
 */

require_once '../config/database.php';
require_once '../config/functions.php';

iniciarSesion();

// Verificar que se haya proporcionado una cédula
if (!isset($_SESSION['registro_cedula'])) {
    redirigir('../index.php');
}

$cedula = trim($_SESSION['registro_cedula']); // Eliminar espacios
$error = '';
$success = false;

// Procesar formulario de registro
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $nombre = sanitizar($_POST['nombre'] ?? '');
    $apellido = sanitizar($_POST['apellido'] ?? '');
    $telefono = sanitizar($_POST['telefono'] ?? '');
    $ciudad = sanitizar($_POST['ciudad'] ?? '');
    
    // Validar campos requeridos
    if (empty($nombre) || empty($apellido) || empty($ciudad)) {
        $error = 'El nombre, apellido y ciudad son obligatorios';
    } else {
        try {
            $db = getDB();
            $stmt = $db->prepare("
                INSERT INTO clientes (cedula_ruc, nombre, apellido, telefono, ciudad) 
                VALUES (?, ?, ?, ?, ?)
            ");
            
            $stmt->execute([trim($cedula), $nombre, $apellido, $telefono, $ciudad]);
            
            // Obtener el ID del cliente recién creado
            $clienteId = $db->lastInsertId();
            
            // Iniciar sesión automáticamente
            $_SESSION['cliente_id'] = $clienteId;
            $_SESSION['cliente_nombre'] = $nombre;
            $_SESSION['cliente_apellido'] = $apellido;
            $_SESSION['cliente_cedula'] = trim($cedula);
            $_SESSION['nuevo_registro'] = true;
            
            unset($_SESSION['registro_cedula']);
            
            redirigir('bienvenida.php');
            
        } catch (PDOException $e) {
            $error = 'Error al registrar el cliente. Intente nuevamente.';
        }
    }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Registro de Cliente - Sistema de Tiendas V3.0</title>
    <link rel="stylesheet" href="../assets/css/style.css">
</head>
<body class="registro-page">
    <div class="registro-container">
        <div class="logo-container">
            <img src="../assets/images/logo.png" alt="Logo Tienda" class="logo" onerror="this.style.display='none'">
            <h1>Registro de Cliente</h1>
        </div>
        
        <div class="registro-card">
            <div class="alert alert-info">
                <strong>¡Bienvenido!</strong> No encontramos su cédula/RUC en nuestro sistema. 
                Por favor complete el siguiente formulario para registrarse.
            </div>
            
            <p class="cedula-info">
                <strong>Cédula/RUC:</strong> <?php echo htmlspecialchars(formatearCedula($cedula)); ?>
            </p>
            
            <?php if ($error): ?>
                <div class="alert alert-error">
                    <?php echo htmlspecialchars($error); ?>
                </div>
            <?php endif; ?>
            
            <form method="POST" action="" class="registro-form">
                <div class="form-row">
                    <div class="form-group">
                        <label for="nombre">Nombre *</label>
                        <input 
                            type="text" 
                            id="nombre" 
                            name="nombre" 
                            class="form-control" 
                            required
                            value="<?php echo htmlspecialchars($_POST['nombre'] ?? ''); ?>"
                        >
                    </div>
                    
                    <div class="form-group">
                        <label for="apellido">Apellido *</label>
                        <input 
                            type="text" 
                            id="apellido" 
                            name="apellido" 
                            class="form-control" 
                            required
                            value="<?php echo htmlspecialchars($_POST['apellido'] ?? ''); ?>"
                        >
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="telefono">Teléfono</label>
                        <input 
                            type="tel" 
                            id="telefono" 
                            name="telefono" 
                            class="form-control"
                            placeholder="Ej: 0981234567"
                            value="<?php echo htmlspecialchars($_POST['telefono'] ?? ''); ?>"
                        >
                    </div>
                    
                    <div class="form-group">
                        <label for="ciudad">Ciudad *</label>
                        <input 
                            type="text" 
                            id="ciudad" 
                            name="ciudad" 
                            class="form-control"
                            placeholder="Ej: Asunción, Ciudad del Este"
                            required
                            value="<?php echo htmlspecialchars($_POST['ciudad'] ?? ''); ?>"
                        >
                    </div>
                </div>
                
                <div class="form-actions">
                    <a href="../index.php" class="btn btn-secondary">Cancelar</a>
                    <button type="submit" class="btn btn-primary">Registrarse</button>
                </div>
            </form>
        </div>
    </div>
    
    <script src="../assets/js/main.js"></script>
</body>
</html>
