<?php
/**
 * Dashboard del cliente - Pantalla principal de compras
 * Sistema de Supermercados y Tiendas V3.0
 */

require_once '../config/database.php';
require_once '../config/functions.php';

iniciarSesion();

// Verificar autenticación
if (!estaAutenticadoCliente()) {
    redirigir('../index.php');
}

$cliente = obtenerClienteActual();
$db = getDB();

// Obtener o crear compra activa
$stmt = $db->prepare("
    SELECT id, total 
    FROM compras 
    WHERE cliente_id = ? AND estado = 'activa' 
    ORDER BY fecha_compra DESC 
    LIMIT 1
");
$stmt->execute([$cliente['id']]);
$compraActiva = $stmt->fetch();

if (!$compraActiva) {
    // Crear nueva compra
    $stmt = $db->prepare("INSERT INTO compras (cliente_id, total, estado) VALUES (?, 0, 'activa')");
    $stmt->execute([$cliente['id']]);
    $compraId = $db->lastInsertId();
} else {
    $compraId = $compraActiva['id'];
}

// Obtener última compra finalizada
$stmt = $db->prepare("
    SELECT c.*, COUNT(dc.id) as total_items
    FROM compras c
    LEFT JOIN detalle_compras dc ON c.id = dc.compra_id
    WHERE c.cliente_id = ? AND c.estado = 'finalizada'
    GROUP BY c.id
    ORDER BY c.fecha_compra DESC
    LIMIT 1
");
$stmt->execute([$cliente['id']]);
$ultimaCompra = $stmt->fetch();

// Obtener promociones activas
$stmt = $db->prepare("
    SELECT * 
    FROM promociones 
    WHERE activo = 1 
    AND (fecha_fin IS NULL OR fecha_fin > NOW())
    ORDER BY fecha_creacion DESC
");
$stmt->execute();
$promociones = $stmt->fetchAll();

// Obtener mensajes emergentes activos
$stmt = $db->prepare("
    SELECT * 
    FROM mensajes_emergentes 
    WHERE activo = 1 
    AND (fecha_expiracion IS NULL OR fecha_expiracion > NOW())
    ORDER BY fecha_creacion DESC
    LIMIT 5
");
$stmt->execute();
$mensajesEmergentes = $stmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - Sistema de Tiendas V3.0</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <script src="https://unpkg.com/html5-qrcode@2.3.8/html5-qrcode.min.js"></script>
</head>
<body class="dashboard-page">
    <!-- Header -->
    <header class="dashboard-header">
        <div class="logo-section">
            <img src="../assets/images/logo.png" alt="Logo" class="logo-small" onerror="this.style.display='none'">
            <span class="store-name">Tienda V3.0</span>
        </div>
        <div class="user-section">
            <span class="user-name">
                Bienvenido, <strong><?php echo htmlspecialchars($cliente['nombre']); ?></strong>
            </span>
            <a href="logout.php" class="btn btn-sm btn-outline">Salir</a>
        </div>
    </header>
    
    <!-- Main Content -->
    <div class="dashboard-container">
        <!-- Panel Izquierdo - Pantalla Principal -->
        <div class="panel-principal">
            <!-- Lector de Código de Barras -->
            <div class="scanner-section">
                <div id="reader" class="scanner-container"></div>
                <button id="btnIniciarScanner" class="btn btn-primary btn-lg">
                    <span class="icon">📷</span>
                    APRIETE EL BOTON PARA LEER EL CODIGO
                </button>
                <div id="scanner-status" class="scanner-status"></div>
                
                <!-- Entrada Manual de Código -->
                <div class="manual-input-section" style="margin-top: 15px;">
                    <p style="text-align: center; color: #666; font-size: 14px; margin-bottom: 10px;">
                        ¿Problemas con la cámara? Ingrese el código manualmente:
                    </p>
                    <div style="display: flex; gap: 10px;">
                        <input type="text" 
                               id="codigoManual" 
                               placeholder="Ej: 7891234567890" 
                               class="form-control"
                               style="flex: 1; padding: 12px; font-size: 16px; border: 2px solid #ddd; border-radius: 8px;"
                               pattern="[0-9]*"
                               inputmode="numeric">
                        <button onclick="buscarPorCodigoManual()" 
                                class="btn btn-success"
                                style="padding: 12px 20px; white-space: nowrap;">
                            🔍 Buscar
                        </button>
                    </div>
                </div>
            </div>
            
            <!-- Producto Escaneado -->
            <div id="producto-preview" class="producto-preview" style="display: none;">
                <div class="producto-imagen">
                    <img id="preview-imagen" src="" alt="Producto">
                </div>
                <div class="producto-info">
                    <h3 id="preview-nombre"></h3>
                    <p id="preview-descripcion"></p>
                    <div class="producto-precio" id="preview-precio"></div>
                </div>
                <div class="producto-acciones">
                    <button class="btn btn-secondary" onclick="rechazarProducto()">No quiero</button>
                    <button class="btn btn-success" onclick="agregarProducto(1)">Si quiero</button>
                    <button class="btn btn-primary" onclick="mostrarCantidad()">Este quiero mas</button>
                </div>
                <div id="cantidad-selector" class="cantidad-selector" style="display: none;">
                    <label>Cantidad:</label>
                    <input type="number" id="cantidad-input" value="2" min="1" max="99">
                    <button class="btn btn-success" onclick="agregarProductoCantidad()">Agregar</button>
                </div>
            </div>
            
            <!-- Área de Promociones -->
            <?php if (!empty($promociones)): ?>
                <div id="promociones-area" class="promociones-area">
                    <?php foreach ($promociones as $index => $promo): ?>
                        <div class="promocion-slide" data-index="<?php echo $index; ?>" style="display: <?php echo $index === 0 ? 'block' : 'none'; ?>">
                            <?php if ($promo['imagen']): ?>
                                <img src="../<?php echo htmlspecialchars($promo['imagen']); ?>" alt="Promoción">
                            <?php endif; ?>
                            <h3><?php echo htmlspecialchars($promo['titulo']); ?></h3>
                            <p><?php echo htmlspecialchars($promo['descripcion']); ?></p>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>
        
        <!-- Panel Derecho - Carrito de Compras -->
        <div class="panel-carrito">
            <div class="carrito-header">
                <h2>🛒 Mi Carrito</h2>
                <span class="items-count" id="items-count">0 items</span>
            </div>
            
            <div class="carrito-items" id="carrito-items">
                <div class="carrito-vacio">
                    <p>Su carrito está vacío</p>
                    <p class="texto-pequeno">Escanee productos para comenzar</p>
                </div>
            </div>
            
            <div class="carrito-footer">
                <div class="total-section">
                    <span class="total-label">TOTAL:</span>
                    <span class="total-monto" id="total-monto">₲ 0</span>
                </div>
                <button class="btn btn-success btn-lg btn-block" id="btn-finalizar" onclick="finalizarCompra()" disabled>
                    Finalizar Compra
                </button>
                <button class="btn btn-danger btn-block" id="btn-vaciar" onclick="vaciarCarrito()" disabled>
                    Vaciar Carrito
                </button>
                <?php if ($ultimaCompra): ?>
                    <button class="btn btn-info btn-block" onclick="verUltimaCompra()" style="margin-top: 10px;">
                        <span class="icon">📋</span>
                        Ver Última Compra
                        <small style="display: block; font-size: 12px;">(<?php echo formatearGuaranies($ultimaCompra['total']); ?> - <?php echo $ultimaCompra['total_items']; ?> items)</small>
                    </button>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <!-- Modal Última Compra -->
    <div id="modal-ultima-compra" class="modal">
        <div class="modal-content">
            <span class="modal-close" onclick="cerrarModalUltimaCompra()">&times;</span>
            <h2>Última Compra</h2>
            <div id="ultima-compra-detalle"></div>
        </div>
    </div>
    
    <!-- Audio para feedback -->
    <audio id="audio-agregar" preload="auto">
        <source src="../assets/sounds/beep.mp3" type="audio/mpeg">
    </audio>
    
    <script>
        const compraId = <?php echo $compraId; ?>;
        let productoActual = null;
        let html5QrcodeScanner = null;
        let scannerActivo = false;
        let promocionIndex = 0;
        const totalPromociones = <?php echo count($promociones); ?>;
        
        // Cargar carrito al iniciar
        document.addEventListener('DOMContentLoaded', function() {
            cargarCarrito();
            iniciarRotacionPromociones();
            mostrarMensajesEmergentes();
        });
        
        // Iniciar/Detener Scanner
        document.getElementById('btnIniciarScanner').addEventListener('click', function() {
            if (scannerActivo) {
                detenerScanner();
            } else {
                iniciarScanner();
            }
        });
        
        function iniciarScanner() {
            const config = {
                fps: 10,
                qrbox: { width: 250, height: 150 },
                aspectRatio: 1.777778,
                // Configuración específica para códigos de barras
                formatsToSupport: [
                    Html5QrcodeSupportedFormats.EAN_13,
                    Html5QrcodeSupportedFormats.EAN_8,
                    Html5QrcodeSupportedFormats.UPC_A,
                    Html5QrcodeSupportedFormats.UPC_E,
                    Html5QrcodeSupportedFormats.CODE_128,
                    Html5QrcodeSupportedFormats.CODE_39,
                    Html5QrcodeSupportedFormats.ITF,
                    Html5QrcodeSupportedFormats.QR_CODE
                ],
                // Configuración avanzada para mejor detección
                experimentalFeatures: {
                    useBarCodeDetectorIfSupported: true
                }
            };
            
            html5QrcodeScanner = new Html5Qrcode("reader");
            
            // Configuración de cámara para iPhone (solo facingMode)
            html5QrcodeScanner.start(
                { facingMode: "environment" },
                config,
                onScanSuccess,
                onScanError
            ).then(() => {
                scannerActivo = true;
                document.getElementById('btnIniciarScanner').textContent = 'DETENER SCANNER';
                document.getElementById('btnIniciarScanner').classList.add('btn-danger');
                document.getElementById('scanner-status').textContent = 'Scanner activo - Apunte al código de barras';
                document.getElementById('scanner-status').style.color = '#4CAF50';
            }).catch(err => {
                console.error('Error al iniciar scanner:', err);
                mostrarMensaje('Error al iniciar la cámara: ' + err, 'error');
                document.getElementById('scanner-status').textContent = 'Error: ' + err;
                document.getElementById('scanner-status').style.color = '#f44336';
            });
        }
        
        function detenerScanner() {
            if (html5QrcodeScanner) {
                html5QrcodeScanner.stop().then(() => {
                    scannerActivo = false;
                    document.getElementById('btnIniciarScanner').textContent = 'APRIETE EL BOTON PARA LEER EL CODIGO';
                    document.getElementById('btnIniciarScanner').classList.remove('btn-danger');
                    document.getElementById('scanner-status').textContent = '';
                }).catch(err => {
                    console.error('Error al detener scanner:', err);
                });
            }
        }
        
        function onScanSuccess(decodedText, decodedResult) {
            console.log('Código escaneado:', decodedText);
            buscarProducto(decodedText);
            detenerScanner();
        }
        
        function onScanError(error) {
            // Ignorar errores de escaneo continuo
        }
        
        // Función para buscar por código manual
        function buscarPorCodigoManual() {
            const codigo = document.getElementById('codigoManual').value.trim();
            
            if (!codigo) {
                alert('⚠️ Por favor ingrese un código de barras');
                return;
            }
            
            if (codigo.length < 6) {
                alert('⚠️ El código debe tener al menos 6 dígitos');
                return;
            }
            
            // Buscar el producto
            buscarProducto(codigo);
            
            // Limpiar el campo
            document.getElementById('codigoManual').value = '';
        }
        
        // Permitir buscar con Enter
        document.addEventListener('DOMContentLoaded', function() {
            document.getElementById('codigoManual').addEventListener('keypress', function(e) {
                if (e.key === 'Enter') {
                    buscarPorCodigoManual();
                }
            });
        });
        
        function buscarProducto(codigoBarras) {
            fetch('api/buscar_producto.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ codigo_barras: codigoBarras })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    mostrarProducto(data.producto);
                } else {
                    mostrarMensaje('Producto no encontrado', 'error');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                mostrarMensaje('Error al buscar el producto', 'error');
            });
        }
        
        function mostrarProducto(producto) {
            productoActual = producto;
            document.getElementById('preview-nombre').textContent = producto.nombre;
            document.getElementById('preview-descripcion').textContent = producto.descripcion || '';
            document.getElementById('preview-precio').textContent = formatearGuaranies(producto.precio);
            
            if (producto.imagen) {
                document.getElementById('preview-imagen').src = '../' + producto.imagen;
            } else {
                document.getElementById('preview-imagen').src = '../assets/images/no-image.png';
            }
            
            document.getElementById('producto-preview').style.display = 'block';
            document.getElementById('promociones-area').style.display = 'none';
        }
        
        function rechazarProducto() {
            ocultarProducto();
        }
        
        function mostrarCantidad() {
            document.getElementById('cantidad-selector').style.display = 'block';
            document.getElementById('cantidad-input').focus();
        }
        
        function agregarProducto(cantidad) {
            if (!productoActual) return;
            
            fetch('api/agregar_carrito.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    compra_id: compraId,
                    producto_id: productoActual.id,
                    cantidad: cantidad
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    reproducirSonido();
                    cargarCarrito();
                    ocultarProducto();
                    mostrarMensaje('Producto agregado al carrito', 'success');
                } else {
                    mostrarMensaje(data.message || 'Error al agregar producto', 'error');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                mostrarMensaje('Error al agregar el producto', 'error');
            });
        }
        
        function agregarProductoCantidad() {
            const cantidad = parseInt(document.getElementById('cantidad-input').value);
            if (cantidad > 0) {
                agregarProducto(cantidad);
            }
        }
        
        function ocultarProducto() {
            document.getElementById('producto-preview').style.display = 'none';
            document.getElementById('cantidad-selector').style.display = 'none';
            document.getElementById('promociones-area').style.display = 'block';
            productoActual = null;
        }
        
        function cargarCarrito() {
            fetch('api/obtener_carrito.php?compra_id=' + compraId)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        actualizarCarrito(data.items, data.total);
                    }
                })
                .catch(error => console.error('Error:', error));
        }
        
        function actualizarCarrito(items, total) {
            const carritoItems = document.getElementById('carrito-items');
            const itemsCount = document.getElementById('items-count');
            const totalMonto = document.getElementById('total-monto');
            const btnFinalizar = document.getElementById('btn-finalizar');
            const btnVaciar = document.getElementById('btn-vaciar');
            
            if (items.length === 0) {
                carritoItems.innerHTML = '<div class="carrito-vacio"><p>Su carrito está vacío</p><p class="texto-pequeno">Escanee productos para comenzar</p></div>';
                btnFinalizar.disabled = true;
                btnVaciar.disabled = true;
            } else {
                let html = '';
                items.forEach(item => {
                    // Construir ruta correcta de la imagen
                    let imagenUrl = '../assets/images/no-image.png';
                    if (item.imagen) {
                        // Si la imagen ya tiene la ruta completa desde la raíz
                        if (item.imagen.startsWith('uploads/')) {
                            imagenUrl = '../' + item.imagen;
                        } else if (item.imagen.startsWith('../')) {
                            imagenUrl = item.imagen;
                        } else {
                            imagenUrl = item.imagen;
                        }
                    }
                    
                    html += `
                        <div class="carrito-item-horizontal">
                            <img src="${imagenUrl}" alt="${item.nombre}" class="item-imagen" onerror="this.src='../assets/images/no-image.png'" onload="console.log('Imagen cargada: ${imagenUrl}')" onerror="console.error('Error cargando imagen: ${imagenUrl}'); this.src='../assets/images/no-image.png'">
                            <div class="item-detalles">
                                <h4 class="item-nombre">${item.nombre}</h4>
                                <p class="item-precio-unit">${formatearGuaranies(item.precio_unitario)} c/u</p>
                            </div>
                            <div class="item-controles">
                                <div class="item-cantidad">
                                    <button class="btn-cantidad" onclick="cambiarCantidad(${item.id}, -1)" title="Disminuir">-</button>
                                    <span class="cantidad">${item.cantidad}</span>
                                    <button class="btn-cantidad" onclick="cambiarCantidad(${item.id}, 1)" title="Aumentar">+</button>
                                </div>
                                <div class="item-subtotal">
                                    ${formatearGuaranies(item.subtotal)}
                                </div>
                            </div>
                            <button class="btn-eliminar" onclick="eliminarItem(${item.id})" title="Eliminar">×</button>
                        </div>
                    `;
                });
                carritoItems.innerHTML = html;
                btnFinalizar.disabled = false;
                btnVaciar.disabled = false;
            }
            
            itemsCount.textContent = items.length + ' items';
            totalMonto.textContent = formatearGuaranies(total);
        }
        
        function cambiarCantidad(detalleId, cambio) {
            fetch('api/actualizar_cantidad.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    detalle_id: detalleId,
                    cambio: cambio
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    cargarCarrito();
                } else {
                    mostrarMensaje(data.message || 'Error al actualizar cantidad', 'error');
                }
            })
            .catch(error => console.error('Error:', error));
        }
        
        function eliminarItem(detalleId) {
            if (confirm('¿Está seguro de eliminar este producto?')) {
                fetch('api/eliminar_item.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ detalle_id: detalleId })
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        cargarCarrito();
                        mostrarMensaje('Producto eliminado', 'success');
                    }
                })
                .catch(error => console.error('Error:', error));
            }
        }
        
        function vaciarCarrito() {
            if (confirm('¿Está seguro de vaciar todo el carrito?')) {
                fetch('api/vaciar_carrito.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ compra_id: compraId })
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        cargarCarrito();
                        mostrarMensaje('Carrito vaciado', 'success');
                    }
                })
                .catch(error => console.error('Error:', error));
            }
        }
        
        function finalizarCompra() {
            if (confirm('¿Confirma que desea finalizar la compra?')) {
                fetch('api/finalizar_compra.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ compra_id: compraId })
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        window.location.href = 'compra_finalizada.php?compra_id=' + compraId;
                    } else {
                        mostrarMensaje(data.message || 'Error al finalizar compra', 'error');
                    }
                })
                .catch(error => console.error('Error:', error));
            }
        }
        
        function verUltimaCompra() {
            fetch('api/obtener_ultima_compra.php')
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        mostrarModalUltimaCompra(data.compra, data.items);
                    }
                })
                .catch(error => console.error('Error:', error));
        }
        
        function mostrarModalUltimaCompra(compra, items) {
            let html = `
                <div class="compra-info">
                    <p><strong>Fecha:</strong> ${compra.fecha_compra}</p>
                    <p><strong>Total:</strong> ${formatearGuaranies(compra.total)}</p>
                </div>
                <table class="tabla-items">
                    <thead>
                        <tr>
                            <th>Producto</th>
                            <th>Cant.</th>
                            <th>Precio</th>
                            <th>Subtotal</th>
                        </tr>
                    </thead>
                    <tbody>
            `;
            
            items.forEach(item => {
                html += `
                    <tr>
                        <td>${item.nombre}</td>
                        <td>${item.cantidad}</td>
                        <td>${formatearGuaranies(item.precio_unitario)}</td>
                        <td>${formatearGuaranies(item.subtotal)}</td>
                    </tr>
                `;
            });
            
            html += '</tbody></table>';
            document.getElementById('ultima-compra-detalle').innerHTML = html;
            document.getElementById('modal-ultima-compra').style.display = 'block';
        }
        
        function cerrarModalUltimaCompra() {
            document.getElementById('modal-ultima-compra').style.display = 'none';
        }
        
        function iniciarRotacionPromociones() {
            if (totalPromociones > 1) {
                setInterval(() => {
                    const slides = document.querySelectorAll('.promocion-slide');
                    slides[promocionIndex].style.display = 'none';
                    promocionIndex = (promocionIndex + 1) % totalPromociones;
                    slides[promocionIndex].style.display = 'block';
                }, 10000); // Cambiar cada 10 segundos
            }
        }
        
        function reproducirSonido() {
            const audio = document.getElementById('audio-agregar');
            audio.play().catch(e => console.log('No se pudo reproducir el sonido'));
        }
        
        function formatearGuaranies(monto) {
            return '₲ ' + Math.round(monto).toLocaleString('es-PY');
        }
        
        function mostrarMensaje(mensaje, tipo) {
            // Solo mostrar mensajes de error importantes, no los de éxito
            console.log(tipo + ': ' + mensaje);
            
            // Solo mostrar alert para errores críticos
            if (tipo === 'error' && !mensaje.includes('Producto no encontrado')) {
                // Crear notificación visual en lugar de alert
                const notif = document.createElement('div');
                notif.className = 'notificacion notificacion-' + tipo;
                notif.textContent = mensaje;
                notif.style.cssText = 'position: fixed; top: 20px; right: 20px; padding: 15px 20px; background: ' + 
                    (tipo === 'error' ? '#f44336' : '#4CAF50') + 
                    '; color: white; border-radius: 8px; z-index: 10000; box-shadow: 0 4px 12px rgba(0,0,0,0.3); animation: slideIn 0.3s ease;';
                document.body.appendChild(notif);
                
                setTimeout(() => {
                    notif.style.animation = 'slideOut 0.3s ease';
                    setTimeout(() => notif.remove(), 300);
                }, 3000);
            }
        }
        
        // Función para mostrar mensajes emergentes
        function mostrarMensajesEmergentes() {
            const mensajes = <?php echo json_encode($mensajesEmergentes); ?>;
            
            if (mensajes.length === 0) return;
            
            let delay = 1000; // Esperar 1 segundo antes del primer mensaje
            
            mensajes.forEach((mensaje, index) => {
                setTimeout(() => {
                    mostrarPopupMensaje(mensaje);
                }, delay + (index * 5000)); // 5 segundos entre cada mensaje
            });
        }
        
        function mostrarPopupMensaje(mensaje) {
            const tipoClass = {
                'oferta': 'popup-oferta',
                'promocion': 'popup-promocion',
                'anuncio': 'popup-anuncio',
                'alerta': 'popup-alerta'
            };
            
            const iconos = {
                'oferta': '🎁',
                'promocion': '🏷️',
                'anuncio': '📢',
                'alerta': '⚠️'
            };
            
            const popup = document.createElement('div');
            popup.className = `mensaje-popup ${tipoClass[mensaje.tipo] || 'popup-anuncio'}`;
            popup.innerHTML = `
                <div class="popup-content">
                    <span class="popup-close" onclick="this.parentElement.parentElement.remove()">&times;</span>
                    <div class="popup-icon">${iconos[mensaje.tipo] || '📢'}</div>
                    <h3>${mensaje.tipo.toUpperCase()}</h3>
                    <p>${mensaje.mensaje}</p>
                    <button class="btn btn-primary btn-sm" onclick="this.parentElement.parentElement.remove()">Entendido</button>
                </div>
            `;
            
            document.body.appendChild(popup);
            
            // Auto-cerrar después de 10 segundos
            setTimeout(() => {
                if (popup.parentElement) {
                    popup.remove();
                }
            }, 10000);
        }
    </script>
    <script src="../assets/js/main.js"></script>
</body>
</html>
