/**
 * JavaScript principal del sistema
 * Sistema de Supermercados y Tiendas V3.0
 */

// Cerrar modales al hacer clic fuera de ellos
window.onclick = function(event) {
    if (event.target.classList.contains('modal')) {
        event.target.style.display = 'none';
    }
}

// Formatear números como moneda guaraní
function formatearGuaranies(monto) {
    return '₲ ' + Math.round(monto).toLocaleString('es-PY');
}

// Validar formato de cédula/RUC
function validarCedulaRuc(cedula) {
    const limpia = cedula.replace(/[^0-9]/g, '');
    return limpia.length >= 6 && limpia.length <= 9;
}

// Mostrar notificación toast (simple)
function mostrarNotificacion(mensaje, tipo = 'info') {
    const colores = {
        success: '#4CAF50',
        error: '#f44336',
        warning: '#ff9800',
        info: '#2196F3'
    };
    
    const notif = document.createElement('div');
    notif.textContent = mensaje;
    notif.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        background-color: ${colores[tipo] || colores.info};
        color: white;
        padding: 15px 25px;
        border-radius: 8px;
        box-shadow: 0 4px 12px rgba(0,0,0,0.2);
        z-index: 10000;
        animation: slideIn 0.3s ease;
    `;
    
    document.body.appendChild(notif);
    
    setTimeout(() => {
        notif.style.animation = 'slideOut 0.3s ease';
        setTimeout(() => notif.remove(), 300);
    }, 3000);
}

// Animaciones CSS
const style = document.createElement('style');
style.textContent = `
    @keyframes slideIn {
        from {
            transform: translateX(400px);
            opacity: 0;
        }
        to {
            transform: translateX(0);
            opacity: 1;
        }
    }
    
    @keyframes slideOut {
        from {
            transform: translateX(0);
            opacity: 1;
        }
        to {
            transform: translateX(400px);
            opacity: 0;
        }
    }
`;
document.head.appendChild(style);

// Confirmar antes de salir si hay cambios sin guardar
let cambiosSinGuardar = false;

window.addEventListener('beforeunload', function(e) {
    if (cambiosSinGuardar) {
        e.preventDefault();
        e.returnValue = '';
    }
});

// Marcar cambios en formularios
document.addEventListener('DOMContentLoaded', function() {
    const formularios = document.querySelectorAll('form');
    formularios.forEach(form => {
        const inputs = form.querySelectorAll('input, textarea, select');
        inputs.forEach(input => {
            input.addEventListener('change', () => {
                cambiosSinGuardar = true;
            });
        });
        
        form.addEventListener('submit', () => {
            cambiosSinGuardar = false;
        });
    });
});

// Autocompletar código de barras con Enter
document.addEventListener('DOMContentLoaded', function() {
    const inputCodigo = document.getElementById('codigo_barras');
    if (inputCodigo) {
        inputCodigo.addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                e.preventDefault();
                // Buscar producto automáticamente
                if (typeof buscarProducto === 'function') {
                    buscarProducto(this.value);
                }
            }
        });
    }
});

// Prevenir envío doble de formularios
document.addEventListener('DOMContentLoaded', function() {
    const formularios = document.querySelectorAll('form');
    formularios.forEach(form => {
        form.addEventListener('submit', function(e) {
            const submitBtn = this.querySelector('button[type="submit"]');
            if (submitBtn) {
                submitBtn.disabled = true;
                setTimeout(() => {
                    submitBtn.disabled = false;
                }, 3000);
            }
        });
    });
});

// Función para copiar texto al portapapeles
function copiarAlPortapapeles(texto) {
    if (navigator.clipboard) {
        navigator.clipboard.writeText(texto).then(() => {
            mostrarNotificacion('Copiado al portapapeles', 'success');
        });
    } else {
        // Fallback para navegadores antiguos
        const textarea = document.createElement('textarea');
        textarea.value = texto;
        document.body.appendChild(textarea);
        textarea.select();
        document.execCommand('copy');
        document.body.removeChild(textarea);
        mostrarNotificacion('Copiado al portapapeles', 'success');
    }
}

// Formatear inputs de precio mientras se escribe
document.addEventListener('DOMContentLoaded', function() {
    const inputsPrecio = document.querySelectorAll('input[name="precio"]');
    inputsPrecio.forEach(input => {
        input.addEventListener('blur', function() {
            if (this.value) {
                const valor = parseFloat(this.value);
                if (!isNaN(valor)) {
                    this.value = valor.toFixed(2);
                }
            }
        });
    });
});

// Validación de archivos de imagen
function validarImagen(input) {
    if (input.files && input.files[0]) {
        const file = input.files[0];
        const tiposPermitidos = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
        const tamañoMaximo = 5 * 1024 * 1024; // 5MB
        
        if (!tiposPermitidos.includes(file.type)) {
            mostrarNotificacion('Tipo de archivo no permitido. Use JPG, PNG, GIF o WEBP.', 'error');
            input.value = '';
            return false;
        }
        
        if (file.size > tamañoMaximo) {
            mostrarNotificacion('El archivo es demasiado grande. Máximo 5MB.', 'error');
            input.value = '';
            return false;
        }
        
        return true;
    }
}

// Aplicar validación a inputs de imagen
document.addEventListener('DOMContentLoaded', function() {
    const inputsImagen = document.querySelectorAll('input[type="file"][accept*="image"]');
    inputsImagen.forEach(input => {
        input.addEventListener('change', function() {
            validarImagen(this);
        });
    });
});

// Previsualización de imágenes
function previsualizarImagen(input, previewId) {
    if (input.files && input.files[0]) {
        const reader = new FileReader();
        reader.onload = function(e) {
            const preview = document.getElementById(previewId);
            if (preview) {
                preview.src = e.target.result;
                preview.style.display = 'block';
            }
        };
        reader.readAsDataURL(input.files[0]);
    }
}

// Búsqueda en tiempo real en tablas
function buscarEnTabla(inputId, tablaId) {
    const input = document.getElementById(inputId);
    const tabla = document.getElementById(tablaId);
    
    if (!input || !tabla) return;
    
    input.addEventListener('keyup', function() {
        const filtro = this.value.toLowerCase();
        const filas = tabla.getElementsByTagName('tr');
        
        for (let i = 1; i < filas.length; i++) {
            const fila = filas[i];
            const texto = fila.textContent || fila.innerText;
            
            if (texto.toLowerCase().indexOf(filtro) > -1) {
                fila.style.display = '';
            } else {
                fila.style.display = 'none';
            }
        }
    });
}

// Confirmar acción
function confirmarAccion(mensaje, callback) {
    if (confirm(mensaje)) {
        callback();
    }
}

// Scroll suave
function scrollSuave(elementId) {
    const element = document.getElementById(elementId);
    if (element) {
        element.scrollIntoView({ behavior: 'smooth', block: 'start' });
    }
}

// Detectar si es dispositivo móvil
function esDispositivoMovil() {
    return /Android|webOS|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent);
}

// Ajustar UI para móviles
document.addEventListener('DOMContentLoaded', function() {
    if (esDispositivoMovil()) {
        document.body.classList.add('mobile-device');
    }
});

// Log de errores en consola (solo en desarrollo)
window.addEventListener('error', function(e) {
    console.error('Error:', e.message, 'en', e.filename, 'línea', e.lineno);
});

// Exportar funciones globales
window.formatearGuaranies = formatearGuaranies;
window.validarCedulaRuc = validarCedulaRuc;
window.mostrarNotificacion = mostrarNotificacion;
window.copiarAlPortapapeles = copiarAlPortapapeles;
window.validarImagen = validarImagen;
window.previsualizarImagen = previsualizarImagen;
window.buscarEnTabla = buscarEnTabla;
window.confirmarAccion = confirmarAccion;
window.scrollSuave = scrollSuave;
window.esDispositivoMovil = esDispositivoMovil;
