/**
 * JavaScript del panel de administración
 * Sistema de Supermercados y Tiendas V3.0
 */

// Toggle sidebar en móviles
function toggleSidebar() {
    const sidebar = document.getElementById('sidebar');
    if (sidebar) {
        sidebar.classList.toggle('active');
    }
}

// Cerrar sidebar al hacer clic fuera en móviles
document.addEventListener('click', function(event) {
    const sidebar = document.getElementById('sidebar');
    const menuToggle = document.querySelector('.menu-toggle');
    
    if (sidebar && sidebar.classList.contains('active')) {
        if (!sidebar.contains(event.target) && event.target !== menuToggle) {
            sidebar.classList.remove('active');
        }
    }
});

// Confirmar eliminación
function confirmarEliminacion(mensaje = '¿Está seguro de eliminar este elemento?') {
    return confirm(mensaje);
}

// Exportar tabla a CSV
function exportarTablaCSV(tablaId, nombreArchivo = 'datos.csv') {
    const tabla = document.getElementById(tablaId);
    if (!tabla) return;
    
    let csv = [];
    const filas = tabla.querySelectorAll('tr');
    
    filas.forEach(fila => {
        const celdas = fila.querySelectorAll('th, td');
        const fila_datos = [];
        celdas.forEach(celda => {
            fila_datos.push('"' + celda.textContent.replace(/"/g, '""') + '"');
        });
        csv.push(fila_datos.join(','));
    });
    
    const csvString = csv.join('\n');
    const blob = new Blob([csvString], { type: 'text/csv;charset=utf-8;' });
    const link = document.createElement('a');
    const url = URL.createObjectURL(blob);
    
    link.setAttribute('href', url);
    link.setAttribute('download', nombreArchivo);
    link.style.visibility = 'hidden';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
}

// Imprimir contenido
function imprimirContenido(elementId) {
    const elemento = document.getElementById(elementId);
    if (!elemento) return;
    
    const ventana = window.open('', '', 'height=600,width=800');
    ventana.document.write('<html><head><title>Imprimir</title>');
    ventana.document.write('<link rel="stylesheet" href="../assets/css/style.css">');
    ventana.document.write('</head><body>');
    ventana.document.write(elemento.innerHTML);
    ventana.document.write('</body></html>');
    ventana.document.close();
    ventana.print();
}

// Actualizar contador de caracteres en textarea
function actualizarContador(textareaId, contadorId, maximo) {
    const textarea = document.getElementById(textareaId);
    const contador = document.getElementById(contadorId);
    
    if (!textarea || !contador) return;
    
    textarea.addEventListener('input', function() {
        const longitud = this.value.length;
        contador.textContent = `${longitud}/${maximo}`;
        
        if (longitud > maximo) {
            contador.style.color = 'red';
        } else {
            contador.style.color = '';
        }
    });
}

// Ordenar tabla por columna
function ordenarTabla(tablaId, columna, ascendente = true) {
    const tabla = document.getElementById(tablaId);
    if (!tabla) return;
    
    const tbody = tabla.querySelector('tbody');
    const filas = Array.from(tbody.querySelectorAll('tr'));
    
    filas.sort((a, b) => {
        const aVal = a.cells[columna].textContent.trim();
        const bVal = b.cells[columna].textContent.trim();
        
        // Intentar comparar como números
        const aNum = parseFloat(aVal.replace(/[^0-9.-]/g, ''));
        const bNum = parseFloat(bVal.replace(/[^0-9.-]/g, ''));
        
        if (!isNaN(aNum) && !isNaN(bNum)) {
            return ascendente ? aNum - bNum : bNum - aNum;
        }
        
        // Comparar como texto
        return ascendente ? 
            aVal.localeCompare(bVal) : 
            bVal.localeCompare(aVal);
    });
    
    // Reordenar filas
    filas.forEach(fila => tbody.appendChild(fila));
}

// Hacer columnas de tabla ordenables
function hacerTablaOrdenable(tablaId) {
    const tabla = document.getElementById(tablaId);
    if (!tabla) return;
    
    const headers = tabla.querySelectorAll('th');
    headers.forEach((header, index) => {
        header.style.cursor = 'pointer';
        header.style.userSelect = 'none';
        
        let ascendente = true;
        header.addEventListener('click', function() {
            ordenarTabla(tablaId, index, ascendente);
            ascendente = !ascendente;
            
            // Indicador visual
            headers.forEach(h => h.classList.remove('ordenado-asc', 'ordenado-desc'));
            this.classList.add(ascendente ? 'ordenado-desc' : 'ordenado-asc');
        });
    });
}

// Paginación de tabla
class TablaPaginada {
    constructor(tablaId, filasPorPagina = 10) {
        this.tabla = document.getElementById(tablaId);
        this.filasPorPagina = filasPorPagina;
        this.paginaActual = 1;
        this.filas = Array.from(this.tabla.querySelectorAll('tbody tr'));
        this.totalPaginas = Math.ceil(this.filas.length / this.filasPorPagina);
        
        this.inicializar();
    }
    
    inicializar() {
        this.crearControles();
        this.mostrarPagina(1);
    }
    
    crearControles() {
        const controles = document.createElement('div');
        controles.className = 'paginacion-controles';
        controles.innerHTML = `
            <button onclick="tablaPaginada.anterior()">← Anterior</button>
            <span id="info-pagina">Página 1 de ${this.totalPaginas}</span>
            <button onclick="tablaPaginada.siguiente()">Siguiente →</button>
        `;
        this.tabla.parentNode.insertBefore(controles, this.tabla.nextSibling);
    }
    
    mostrarPagina(numero) {
        this.paginaActual = numero;
        const inicio = (numero - 1) * this.filasPorPagina;
        const fin = inicio + this.filasPorPagina;
        
        this.filas.forEach((fila, index) => {
            fila.style.display = (index >= inicio && index < fin) ? '' : 'none';
        });
        
        document.getElementById('info-pagina').textContent = 
            `Página ${numero} de ${this.totalPaginas}`;
    }
    
    siguiente() {
        if (this.paginaActual < this.totalPaginas) {
            this.mostrarPagina(this.paginaActual + 1);
        }
    }
    
    anterior() {
        if (this.paginaActual > 1) {
            this.mostrarPagina(this.paginaActual - 1);
        }
    }
}

// Validación de formularios
function validarFormulario(formId) {
    const form = document.getElementById(formId);
    if (!form) return false;
    
    const inputs = form.querySelectorAll('[required]');
    let valido = true;
    
    inputs.forEach(input => {
        if (!input.value.trim()) {
            input.style.borderColor = 'red';
            valido = false;
        } else {
            input.style.borderColor = '';
        }
    });
    
    return valido;
}

// Guardar estado de filtros en localStorage
function guardarFiltros(clave, valores) {
    localStorage.setItem(clave, JSON.stringify(valores));
}

function cargarFiltros(clave) {
    const datos = localStorage.getItem(clave);
    return datos ? JSON.parse(datos) : null;
}

// Autoguardado de formularios
function activarAutoguardado(formId, clave) {
    const form = document.getElementById(formId);
    if (!form) return;
    
    // Cargar datos guardados
    const datosGuardados = localStorage.getItem(clave);
    if (datosGuardados) {
        const datos = JSON.parse(datosGuardados);
        Object.keys(datos).forEach(name => {
            const input = form.querySelector(`[name="${name}"]`);
            if (input) input.value = datos[name];
        });
    }
    
    // Guardar al cambiar
    form.addEventListener('input', function() {
        const formData = new FormData(form);
        const datos = {};
        formData.forEach((value, key) => {
            datos[key] = value;
        });
        localStorage.setItem(clave, JSON.stringify(datos));
    });
    
    // Limpiar al enviar
    form.addEventListener('submit', function() {
        localStorage.removeItem(clave);
    });
}

// Gráfico de línea simple (sin Chart.js)
function crearGraficoSimple(canvasId, datos, etiquetas) {
    const canvas = document.getElementById(canvasId);
    if (!canvas) return;
    
    const ctx = canvas.getContext('2d');
    const width = canvas.width;
    const height = canvas.height;
    const padding = 40;
    
    // Limpiar canvas
    ctx.clearRect(0, 0, width, height);
    
    // Calcular escalas
    const maxValor = Math.max(...datos);
    const escalaY = (height - 2 * padding) / maxValor;
    const escalaX = (width - 2 * padding) / (datos.length - 1);
    
    // Dibujar ejes
    ctx.strokeStyle = '#ccc';
    ctx.beginPath();
    ctx.moveTo(padding, padding);
    ctx.lineTo(padding, height - padding);
    ctx.lineTo(width - padding, height - padding);
    ctx.stroke();
    
    // Dibujar línea
    ctx.strokeStyle = '#4CAF50';
    ctx.lineWidth = 2;
    ctx.beginPath();
    
    datos.forEach((valor, index) => {
        const x = padding + index * escalaX;
        const y = height - padding - valor * escalaY;
        
        if (index === 0) {
            ctx.moveTo(x, y);
        } else {
            ctx.lineTo(x, y);
        }
    });
    
    ctx.stroke();
    
    // Dibujar puntos
    ctx.fillStyle = '#4CAF50';
    datos.forEach((valor, index) => {
        const x = padding + index * escalaX;
        const y = height - padding - valor * escalaY;
        ctx.beginPath();
        ctx.arc(x, y, 4, 0, 2 * Math.PI);
        ctx.fill();
    });
}

// Inicialización al cargar la página
document.addEventListener('DOMContentLoaded', function() {
    // Hacer tablas ordenables
    const tablas = document.querySelectorAll('.data-table');
    tablas.forEach(tabla => {
        if (tabla.id) {
            hacerTablaOrdenable(tabla.id);
        }
    });
    
    // Tooltip simple
    const elementos = document.querySelectorAll('[data-tooltip]');
    elementos.forEach(elemento => {
        elemento.addEventListener('mouseenter', function() {
            const tooltip = document.createElement('div');
            tooltip.className = 'tooltip-custom';
            tooltip.textContent = this.dataset.tooltip;
            tooltip.style.cssText = `
                position: absolute;
                background: #333;
                color: white;
                padding: 5px 10px;
                border-radius: 4px;
                font-size: 12px;
                z-index: 10000;
                pointer-events: none;
            `;
            document.body.appendChild(tooltip);
            
            const rect = this.getBoundingClientRect();
            tooltip.style.left = rect.left + 'px';
            tooltip.style.top = (rect.top - tooltip.offsetHeight - 5) + 'px';
            
            this._tooltip = tooltip;
        });
        
        elemento.addEventListener('mouseleave', function() {
            if (this._tooltip) {
                this._tooltip.remove();
                this._tooltip = null;
            }
        });
    });
});

// Exportar funciones globales
window.toggleSidebar = toggleSidebar;
window.confirmarEliminacion = confirmarEliminacion;
window.exportarTablaCSV = exportarTablaCSV;
window.imprimirContenido = imprimirContenido;
window.ordenarTabla = ordenarTabla;
window.hacerTablaOrdenable = hacerTablaOrdenable;
window.TablaPaginada = TablaPaginada;
window.validarFormulario = validarFormulario;
window.guardarFiltros = guardarFiltros;
window.cargarFiltros = cargarFiltros;
window.activarAutoguardado = activarAutoguardado;
