<?php
/**
 * Gestión de promociones
 * Sistema de Supermercados y Tiendas V3.0
 */

require_once '../config/database.php';
require_once '../config/functions.php';

iniciarSesion();

if (!estaAutenticadoAdmin()) {
    redirigir('login.php');
}

if (!tienePermiso('gestionar_promociones')) {
    redirigir('dashboard.php');
}

$admin = obtenerAdminActual();
$db = getDB();

$stmt = $db->query("
    SELECT p.*, u.nombre as creado_por_nombre
    FROM promociones p
    LEFT JOIN usuarios_admin u ON p.creado_por = u.id
    ORDER BY p.fecha_creacion DESC
");
$promociones = $stmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Promociones - Panel de Administración</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="../assets/css/admin.css">
</head>
<body class="admin-page">
    <?php include 'includes/header.php'; ?>
    <?php include 'includes/sidebar.php'; ?>
    
    <main class="admin-content">
        <div class="content-header">
            <h1>Gestión de Promociones</h1>
            <button class="btn btn-primary" onclick="abrirModalPromocion()">
                ➕ Nueva Promoción
            </button>
        </div>
        
        <div class="card">
            <div class="card-body">
                <div class="table-responsive">
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>Título</th>
                                <th>Descripción</th>
                                <th>Fecha Inicio</th>
                                <th>Fecha Fin</th>
                                <th>Estado</th>
                                <th>Creado por</th>
                                <th>Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($promociones as $promo): ?>
                                <tr>
                                    <td><strong><?php echo htmlspecialchars($promo['titulo']); ?></strong></td>
                                    <td><?php echo htmlspecialchars(substr($promo['descripcion'], 0, 50)); ?>...</td>
                                    <td><?php echo date('d/m/Y', strtotime($promo['fecha_inicio'])); ?></td>
                                    <td><?php echo $promo['fecha_fin'] ? date('d/m/Y', strtotime($promo['fecha_fin'])) : 'Sin límite'; ?></td>
                                    <td>
                                        <span class="badge badge-<?php echo $promo['activo'] ? 'success' : 'secondary'; ?>">
                                            <?php echo $promo['activo'] ? 'Activa' : 'Inactiva'; ?>
                                        </span>
                                    </td>
                                    <td><?php echo htmlspecialchars($promo['creado_por_nombre'] ?? 'N/A'); ?></td>
                                    <td class="actions-cell">
                                        <button class="btn btn-sm btn-info" onclick='editarPromocion(<?php echo json_encode($promo); ?>)'>✏️</button>
                                        <button class="btn btn-sm btn-danger" onclick="eliminarPromocion(<?php echo $promo['id']; ?>)">🗑️</button>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </main>
    
    <div id="modalPromocion" class="modal">
        <div class="modal-content modal-lg">
            <span class="modal-close" onclick="cerrarModalPromocion()">&times;</span>
            <h2 id="modalTitulo">Nueva Promoción</h2>
            
            <form id="formPromocion" onsubmit="guardarPromocion(event)">
                <input type="hidden" id="promocion_id" name="id">
                
                <div class="form-group">
                    <label for="titulo">Título *</label>
                    <input type="text" id="titulo" name="titulo" class="form-control" required>
                </div>
                
                <div class="form-group">
                    <label for="descripcion">Descripción</label>
                    <textarea id="descripcion" name="descripcion" class="form-control" rows="4"></textarea>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="fecha_inicio">Fecha Inicio</label>
                        <input type="datetime-local" id="fecha_inicio" name="fecha_inicio" class="form-control">
                    </div>
                    
                    <div class="form-group">
                        <label for="fecha_fin">Fecha Fin</label>
                        <input type="datetime-local" id="fecha_fin" name="fecha_fin" class="form-control">
                    </div>
                </div>
                
                <div class="form-group">
                    <label for="imagen">Imagen</label>
                    <input type="file" id="imagen" name="imagen" class="form-control" accept="image/*">
                </div>
                
                <div class="form-group">
                    <label class="checkbox-label">
                        <input type="checkbox" id="activo" name="activo" checked>
                        Promoción activa
                    </label>
                </div>
                
                <div class="form-actions">
                    <button type="button" class="btn btn-secondary" onclick="cerrarModalPromocion()">Cancelar</button>
                    <button type="submit" class="btn btn-primary">Guardar</button>
                </div>
            </form>
        </div>
    </div>
    
    <script>
        function abrirModalPromocion() {
            document.getElementById('modalTitulo').textContent = 'Nueva Promoción';
            document.getElementById('formPromocion').reset();
            document.getElementById('promocion_id').value = '';
            document.getElementById('modalPromocion').style.display = 'block';
        }
        
        function cerrarModalPromocion() {
            document.getElementById('modalPromocion').style.display = 'none';
        }
        
        function editarPromocion(promo) {
            document.getElementById('modalTitulo').textContent = 'Editar Promoción';
            document.getElementById('promocion_id').value = promo.id;
            document.getElementById('titulo').value = promo.titulo;
            document.getElementById('descripcion').value = promo.descripcion || '';
            document.getElementById('activo').checked = promo.activo == 1;
            document.getElementById('modalPromocion').style.display = 'block';
        }
        
        function guardarPromocion(event) {
            event.preventDefault();
            
            const formData = new FormData(document.getElementById('formPromocion'));
            formData.append('activo', document.getElementById('activo').checked ? 1 : 0);
            
            fetch('api/guardar_promocion.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert('Promoción guardada exitosamente');
                    location.reload();
                } else {
                    alert('Error: ' + (data.message || 'No se pudo guardar'));
                }
            });
        }
        
        function eliminarPromocion(id) {
            if (confirm('¿Está seguro de eliminar esta promoción?')) {
                fetch('api/eliminar_promocion.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ id: id })
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        alert('Promoción eliminada');
                        location.reload();
                    }
                });
            }
        }
    </script>
    <script src="../assets/js/admin.js"></script>
</body>
</html>
