<?php
/**
 * Gestión de productos
 * Sistema de Supermercados y Tiendas V3.0
 */

require_once '../config/database.php';
require_once '../config/functions.php';

iniciarSesion();

if (!estaAutenticadoAdmin()) {
    redirigir('login.php');
}

$admin = obtenerAdminActual();
$db = getDB();

// Obtener todos los productos
$stmt = $db->query("
    SELECT 
        p.*,
        u.nombre as creado_por_nombre
    FROM productos p
    LEFT JOIN usuarios_admin u ON p.creado_por = u.id
    ORDER BY p.fecha_creacion DESC
");
$productos = $stmt->fetchAll();

// Obtener categorías únicas
$stmt = $db->query("SELECT DISTINCT categoria FROM productos WHERE categoria IS NOT NULL ORDER BY categoria");
$categorias = $stmt->fetchAll(PDO::FETCH_COLUMN);
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Productos - Panel de Administración</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="../assets/css/admin.css">
</head>
<body class="admin-page">
    <?php include 'includes/header.php'; ?>
    <?php include 'includes/sidebar.php'; ?>
    
    <main class="admin-content">
        <div class="content-header">
            <h1>Gestión de Productos</h1>
            <?php if (tienePermiso('crear')): ?>
                <button class="btn btn-primary" onclick="abrirModalProducto()">
                    ➕ Nuevo Producto
                </button>
            <?php endif; ?>
        </div>
        
        <!-- Filtros -->
        <div class="card">
            <div class="card-body">
                <div class="filters-row">
                    <input type="text" id="buscar" class="form-control" placeholder="Buscar por nombre o código de barras..." onkeyup="filtrarProductos()">
                    <select id="filtroCategoria" class="form-control" onchange="filtrarProductos()">
                        <option value="">Todas las categorías</option>
                        <?php foreach ($categorias as $cat): ?>
                            <option value="<?php echo htmlspecialchars($cat); ?>"><?php echo htmlspecialchars($cat); ?></option>
                        <?php endforeach; ?>
                    </select>
                    <select id="filtroEstado" class="form-control" onchange="filtrarProductos()">
                        <option value="">Todos los estados</option>
                        <option value="1">Activos</option>
                        <option value="0">Inactivos</option>
                    </select>
                </div>
            </div>
        </div>
        
        <!-- Tabla de productos -->
        <div class="card">
            <div class="card-body">
                <div class="table-responsive">
                    <table class="data-table" id="tablaProductos">
                        <thead>
                            <tr>
                                <th>Código</th>
                                <th>Nombre</th>
                                <th>Categoría</th>
                                <th>Precio</th>
                                <th>Stock</th>
                                <th>Estado</th>
                                <th>Creado por</th>
                                <th>Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($productos as $prod): ?>
                                <tr data-categoria="<?php echo htmlspecialchars($prod['categoria']); ?>" 
                                    data-estado="<?php echo $prod['activo']; ?>"
                                    data-nombre="<?php echo htmlspecialchars(strtolower($prod['nombre'])); ?>"
                                    data-codigo="<?php echo htmlspecialchars($prod['codigo_barras']); ?>">
                                    <td><?php echo htmlspecialchars($prod['codigo_barras']); ?></td>
                                    <td>
                                        <strong><?php echo htmlspecialchars($prod['nombre']); ?></strong>
                                        <?php if ($prod['descripcion']): ?>
                                            <br><small><?php echo htmlspecialchars(substr($prod['descripcion'], 0, 50)); ?>...</small>
                                        <?php endif; ?>
                                    </td>
                                    <td><?php echo htmlspecialchars($prod['categoria']); ?></td>
                                    <td><?php echo formatearGuaranies($prod['precio']); ?></td>
                                    <td><?php echo $prod['stock']; ?></td>
                                    <td>
                                        <span class="badge badge-<?php echo $prod['activo'] ? 'success' : 'secondary'; ?>">
                                            <?php echo $prod['activo'] ? 'Activo' : 'Inactivo'; ?>
                                        </span>
                                    </td>
                                    <td><?php echo htmlspecialchars($prod['creado_por_nombre'] ?? 'N/A'); ?></td>
                                    <td class="actions-cell">
                                        <?php if (tienePermiso('modificar') || ($admin['rol'] === 'colaborador' && $prod['creado_por'] == $admin['id'])): ?>
                                            <button class="btn btn-sm btn-info" onclick='editarProducto(<?php echo json_encode($prod); ?>)'>
                                                ✏️
                                            </button>
                                        <?php endif; ?>
                                        
                                        <?php if (tienePermiso('eliminar')): ?>
                                            <button class="btn btn-sm btn-danger" onclick="eliminarProducto(<?php echo $prod['id']; ?>)">
                                                🗑️
                                            </button>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </main>
    
    <!-- Modal Producto -->
    <div id="modalProducto" class="modal">
        <div class="modal-content modal-lg">
            <span class="modal-close" onclick="cerrarModalProducto()">&times;</span>
            <h2 id="modalTitulo">Nuevo Producto</h2>
            
            <form id="formProducto" onsubmit="guardarProducto(event)">
                <input type="hidden" id="producto_id" name="id">
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="codigo_barras">Código de Barras *</label>
                        <div style="display: flex; gap: 10px; align-items: flex-end;">
                            <input type="text" id="codigo_barras" name="codigo_barras" class="form-control" required style="flex: 1;">
                            <button type="button" class="btn btn-success" onclick="abrirScannerAdmin()" title="Escanear código de barras">
                                📷 Escanear
                            </button>
                        </div>
                        <small class="form-text">Puede escribir el código o usar el botón para escanearlo con la cámara</small>
                    </div>
                    
                    <div class="form-group">
                        <label for="categoria">Categoría</label>
                        <input type="text" id="categoria" name="categoria" class="form-control" list="listaCategorias">
                        <datalist id="listaCategorias">
                            <?php foreach ($categorias as $cat): ?>
                                <option value="<?php echo htmlspecialchars($cat); ?>">
                            <?php endforeach; ?>
                        </datalist>
                    </div>
                </div>
                
                <div class="form-group">
                    <label for="nombre">Nombre del Producto *</label>
                    <input type="text" id="nombre" name="nombre" class="form-control" required>
                </div>
                
                <div class="form-group">
                    <label for="descripcion">Descripción</label>
                    <textarea id="descripcion" name="descripcion" class="form-control" rows="3"></textarea>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="precio">Precio (₲) *</label>
                        <input type="text" id="precio" name="precio" class="form-control" required placeholder="Ej: 12.500">
                        <input type="hidden" id="precio_numerico" name="precio_numerico">
                    </div>
                    
                    <div class="form-group">
                        <label for="stock">Stock</label>
                        <input type="number" id="stock" name="stock" class="form-control" value="0" min="0">
                    </div>
                </div>
                
                <div class="form-group">
                    <label for="imagen">Imagen del Producto</label>
                    <input type="file" id="imagen" name="imagen" class="form-control" accept="image/*">
                    <small class="form-text">Formatos permitidos: JPG, PNG, GIF, WEBP (máx. 5MB)</small>
                </div>
                
                <div class="form-group">
                    <label class="checkbox-label">
                        <input type="checkbox" id="activo" name="activo" checked>
                        Producto activo
                    </label>
                </div>
                
                <div class="form-actions">
                    <button type="button" class="btn btn-secondary" onclick="cerrarModalProducto()">Cancelar</button>
                    <button type="submit" class="btn btn-primary">Guardar Producto</button>
                </div>
            </form>
        </div>
    </div>
    
    <script>
        function abrirModalProducto() {
            document.getElementById('modalTitulo').textContent = 'Nuevo Producto';
            document.getElementById('formProducto').reset();
            document.getElementById('producto_id').value = '';
            document.getElementById('modalProducto').style.display = 'block';
        }
        
        function cerrarModalProducto() {
            document.getElementById('modalProducto').style.display = 'none';
        }
        
        // Formato de miles automático para el precio
        function formatearMiles(valor) {
            // Eliminar todo excepto números
            const numero = valor.replace(/[^0-9]/g, '');
            // Agregar puntos de miles
            return numero.replace(/\B(?=(\d{3})+(?!\d))/g, '.');
        }
        
        function quitarFormatoMiles(valor) {
            return valor.replace(/\./g, '');
        }
        
        // Aplicar formato al escribir
        document.getElementById('precio').addEventListener('input', function(e) {
            const cursorPos = this.selectionStart;
            const valorAnterior = this.value;
            const valorSinFormato = quitarFormatoMiles(this.value);
            const valorFormateado = formatearMiles(valorSinFormato);
            
            this.value = valorFormateado;
            document.getElementById('precio_numerico').value = valorSinFormato;
            
            // Ajustar posición del cursor
            const diferencia = valorFormateado.length - valorAnterior.length;
            this.setSelectionRange(cursorPos + diferencia, cursorPos + diferencia);
        });
        
        function editarProducto(producto) {
            document.getElementById('modalTitulo').textContent = 'Editar Producto';
            document.getElementById('producto_id').value = producto.id;
            document.getElementById('codigo_barras').value = producto.codigo_barras;
            document.getElementById('nombre').value = producto.nombre;
            document.getElementById('descripcion').value = producto.descripcion || '';
            // Formatear el precio al cargar
            const precioFormateado = formatearMiles(String(Math.round(producto.precio)));
            document.getElementById('precio').value = precioFormateado;
            document.getElementById('precio_numerico').value = Math.round(producto.precio);
            document.getElementById('stock').value = producto.stock;
            document.getElementById('categoria').value = producto.categoria || '';
            document.getElementById('activo').checked = producto.activo == 1;
            document.getElementById('modalProducto').style.display = 'block';
        }
        
        function guardarProducto(event) {
            event.preventDefault();
            
            // Obtener el precio sin formato
            const precioSinFormato = quitarFormatoMiles(document.getElementById('precio').value);
            
            const formData = new FormData(document.getElementById('formProducto'));
            formData.set('precio', precioSinFormato); // Enviar precio sin puntos
            formData.append('activo', document.getElementById('activo').checked ? 1 : 0);
            
            fetch('api/guardar_producto.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert('Producto guardado exitosamente');
                    location.reload();
                } else {
                    alert('Error: ' + (data.message || 'No se pudo guardar el producto'));
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Error al guardar el producto');
            });
        }
        
        function eliminarProducto(id) {
            if (confirm('¿Está seguro de eliminar este producto?')) {
                fetch('api/eliminar_producto.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ id: id })
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        alert('Producto eliminado exitosamente');
                        location.reload();
                    } else {
                        alert('Error: ' + (data.message || 'No se pudo eliminar el producto'));
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Error al eliminar el producto');
                });
            }
        }
        
        function filtrarProductos() {
            const buscar = document.getElementById('buscar').value.toLowerCase();
            const categoria = document.getElementById('filtroCategoria').value;
            const estado = document.getElementById('filtroEstado').value;
            
            const filas = document.querySelectorAll('#tablaProductos tbody tr');
            
            filas.forEach(fila => {
                const nombre = fila.dataset.nombre;
                const codigo = fila.dataset.codigo;
                const filaCat = fila.dataset.categoria;
                const filaEstado = fila.dataset.estado;
                
                let mostrar = true;
                
                if (buscar && !nombre.includes(buscar) && !codigo.includes(buscar)) {
                    mostrar = false;
                }
                
                if (categoria && filaCat !== categoria) {
                    mostrar = false;
                }
                
                if (estado !== '' && filaEstado !== estado) {
                    mostrar = false;
                }
                
                fila.style.display = mostrar ? '' : 'none';
            });
        }
    </script>
    
    <!-- Modal Scanner de Código de Barras -->
    <div id="modalScanner" class="modal">
        <div class="modal-content">
            <span class="modal-close" onclick="cerrarScannerAdmin()">&times;</span>
            <h2>Escanear Código de Barras</h2>
            
            <div id="readerAdmin" style="width: 100%; max-width: 500px; margin: 20px auto;"></div>
            
            <div style="text-align: center; margin-top: 20px;">
                <p id="scanner-status-admin" style="color: #666; margin-bottom: 15px;"></p>
                <button type="button" class="btn btn-secondary" onclick="cerrarScannerAdmin()">
                    Cancelar
                </button>
            </div>
        </div>
    </div>
    
    <!-- Librería Html5-QRCode -->
    <script src="https://unpkg.com/html5-qrcode@2.3.8/html5-qrcode.min.js"></script>
    
    <script>
        let html5QrcodeScannerAdmin = null;
        
        function abrirScannerAdmin() {
            document.getElementById('modalScanner').style.display = 'block';
            iniciarScannerAdmin();
        }
        
        function cerrarScannerAdmin() {
            if (html5QrcodeScannerAdmin) {
                html5QrcodeScannerAdmin.stop().then(() => {
                    html5QrcodeScannerAdmin = null;
                    document.getElementById('modalScanner').style.display = 'none';
                }).catch(err => {
                    console.error('Error al detener scanner:', err);
                    document.getElementById('modalScanner').style.display = 'none';
                });
            } else {
                document.getElementById('modalScanner').style.display = 'none';
            }
        }
        
        function iniciarScannerAdmin() {
            const config = {
                fps: 10,
                qrbox: { width: 250, height: 150 },
                aspectRatio: 1.777778,
                formatsToSupport: [
                    Html5QrcodeSupportedFormats.EAN_13,
                    Html5QrcodeSupportedFormats.EAN_8,
                    Html5QrcodeSupportedFormats.UPC_A,
                    Html5QrcodeSupportedFormats.UPC_E,
                    Html5QrcodeSupportedFormats.CODE_128,
                    Html5QrcodeSupportedFormats.CODE_39,
                    Html5QrcodeSupportedFormats.ITF,
                    Html5QrcodeSupportedFormats.QR_CODE
                ],
                experimentalFeatures: {
                    useBarCodeDetectorIfSupported: true
                }
            };
            
            html5QrcodeScannerAdmin = new Html5Qrcode("readerAdmin");
            
            html5QrcodeScannerAdmin.start(
                { facingMode: "environment" },
                config,
                onScanSuccessAdmin,
                onScanErrorAdmin
            ).then(() => {
                document.getElementById('scanner-status-admin').textContent = 'Scanner activo - Apunte al código de barras';
                document.getElementById('scanner-status-admin').style.color = '#4CAF50';
            }).catch(err => {
                console.error('Error al iniciar scanner:', err);
                document.getElementById('scanner-status-admin').textContent = 'Error al iniciar la cámara: ' + err;
                document.getElementById('scanner-status-admin').style.color = '#f44336';
            });
        }
        
        function onScanSuccessAdmin(decodedText, decodedResult) {
            console.log('Código escaneado:', decodedText);
            
            // Colocar el código en el campo
            document.getElementById('codigo_barras').value = decodedText;
            
            // Mostrar mensaje de éxito
            document.getElementById('scanner-status-admin').textContent = '✅ Código detectado: ' + decodedText;
            document.getElementById('scanner-status-admin').style.color = '#4CAF50';
            
            // Cerrar el scanner después de 1 segundo
            setTimeout(() => {
                cerrarScannerAdmin();
            }, 1000);
        }
        
        function onScanErrorAdmin(error) {
            // Ignorar errores de escaneo continuo
        }
    </script>
    
    <script src="../assets/js/admin.js"></script>
</body>
</html>
