<?php
/**
 * Gestión de compras/ventas
 * Sistema de Supermercados y Tiendas V3.0
 */

require_once '../config/database.php';
require_once '../config/functions.php';

iniciarSesion();

if (!estaAutenticadoAdmin()) {
    redirigir('login.php');
}

if (!tienePermiso('ver_compras')) {
    redirigir('dashboard.php');
}

$admin = obtenerAdminActual();
$db = getDB();

// Filtros
$filtroEstado = $_GET['estado'] ?? 'todas';
$filtroFecha = $_GET['fecha'] ?? 'hoy';

// Construir query según filtros
$whereConditions = [];
$params = [];

if ($filtroEstado !== 'todas') {
    $whereConditions[] = "c.estado = ?";
    $params[] = $filtroEstado;
}

switch ($filtroFecha) {
    case 'hoy':
        $whereConditions[] = "DATE(c.fecha_compra) = CURDATE()";
        break;
    case 'semana':
        $whereConditions[] = "c.fecha_compra >= DATE_SUB(NOW(), INTERVAL 7 DAY)";
        break;
    case 'mes':
        $whereConditions[] = "c.fecha_compra >= DATE_SUB(NOW(), INTERVAL 30 DAY)";
        break;
}

$whereClause = !empty($whereConditions) ? 'WHERE ' . implode(' AND ', $whereConditions) : '';

// Obtener compras
$query = "
    SELECT 
        c.*,
        cl.nombre,
        cl.apellido,
        cl.cedula_ruc,
        COUNT(dc.id) as num_items
    FROM compras c
    JOIN clientes cl ON c.cliente_id = cl.id
    LEFT JOIN detalle_compras dc ON c.id = dc.compra_id
    $whereClause
    GROUP BY c.id
    ORDER BY c.fecha_compra DESC
    LIMIT 100
";

$stmt = $db->prepare($query);
$stmt->execute($params);
$compras = $stmt->fetchAll();

// Estadísticas del día
$statsQuery = "
    SELECT 
        COUNT(*) as total_compras,
        SUM(total) as total_ventas,
        AVG(total) as promedio_compra
    FROM compras
    WHERE DATE(fecha_compra) = CURDATE() AND estado = 'finalizada'
";
$stats = $db->query($statsQuery)->fetch();
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Compras/Ventas - Panel de Administración</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="../assets/css/admin.css">
</head>
<body class="admin-page">
    <?php include 'includes/header.php'; ?>
    <?php include 'includes/sidebar.php'; ?>
    
    <main class="admin-content">
        <div class="content-header">
            <h1>Gestión de Compras/Ventas</h1>
        </div>
        
        <!-- Estadísticas del día -->
        <div class="stats-grid">
            <div class="stat-card stat-primary">
                <div class="stat-icon">🛒</div>
                <div class="stat-info">
                    <h3><?php echo $stats['total_compras'] ?? 0; ?></h3>
                    <p>Compras Hoy</p>
                </div>
            </div>
            
            <div class="stat-card stat-success">
                <div class="stat-icon">💰</div>
                <div class="stat-info">
                    <h3><?php echo formatearGuaranies($stats['total_ventas'] ?? 0); ?></h3>
                    <p>Ventas Hoy</p>
                </div>
            </div>
            
            <div class="stat-card stat-info">
                <div class="stat-icon">📊</div>
                <div class="stat-info">
                    <h3><?php echo formatearGuaranies($stats['promedio_compra'] ?? 0); ?></h3>
                    <p>Promedio por Compra</p>
                </div>
            </div>
        </div>
        
        <!-- Filtros -->
        <div class="card">
            <div class="card-body">
                <form method="GET" action="" class="filters-form">
                    <div class="form-row">
                        <div class="form-group">
                            <label for="estado">Estado</label>
                            <select id="estado" name="estado" class="form-control">
                                <option value="todas" <?php echo $filtroEstado === 'todas' ? 'selected' : ''; ?>>Todas</option>
                                <option value="finalizada" <?php echo $filtroEstado === 'finalizada' ? 'selected' : ''; ?>>Finalizadas</option>
                                <option value="activa" <?php echo $filtroEstado === 'activa' ? 'selected' : ''; ?>>Activas</option>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label for="fecha">Período</label>
                            <select id="fecha" name="fecha" class="form-control">
                                <option value="hoy" <?php echo $filtroFecha === 'hoy' ? 'selected' : ''; ?>>Hoy</option>
                                <option value="semana" <?php echo $filtroFecha === 'semana' ? 'selected' : ''; ?>>Última semana</option>
                                <option value="mes" <?php echo $filtroFecha === 'mes' ? 'selected' : ''; ?>>Último mes</option>
                                <option value="todas" <?php echo $filtroFecha === 'todas' ? 'selected' : ''; ?>>Todas</option>
                            </select>
                        </div>
                        
                        <div class="form-group" style="align-self: flex-end;">
                            <button type="submit" class="btn btn-primary">Filtrar</button>
                            <a href="compras.php" class="btn btn-secondary">Limpiar</a>
                        </div>
                    </div>
                </form>
            </div>
        </div>
        
        <!-- Tabla de compras -->
        <div class="card">
            <div class="card-header">
                <h2>Listado de Compras (Últimas 100)</h2>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Fecha y Hora</th>
                                <th>Cliente</th>
                                <th>Cédula/RUC</th>
                                <th>Items</th>
                                <th>Total</th>
                                <th>Estado</th>
                                <th>Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($compras)): ?>
                                <tr>
                                    <td colspan="8" class="text-center">No hay compras registradas</td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($compras as $compra): ?>
                                    <tr>
                                        <td><strong>#<?php echo $compra['id']; ?></strong></td>
                                        <td><?php echo date('d/m/Y H:i', strtotime($compra['fecha_compra'])); ?></td>
                                        <td><?php echo htmlspecialchars($compra['nombre'] . ' ' . $compra['apellido']); ?></td>
                                        <td><?php echo htmlspecialchars(formatearCedula($compra['cedula_ruc'])); ?></td>
                                        <td><?php echo $compra['num_items']; ?> items</td>
                                        <td><strong><?php echo formatearGuaranies($compra['total']); ?></strong></td>
                                        <td>
                                            <span class="badge badge-<?php echo $compra['estado'] === 'finalizada' ? 'success' : 'warning'; ?>">
                                                <?php echo ucfirst($compra['estado']); ?>
                                            </span>
                                        </td>
                                        <td class="actions-cell">
                                            <button class="btn btn-sm btn-info" onclick='verDetalleCompra(<?php echo $compra['id']; ?>)'>
                                                👁️ Ver
                                            </button>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </main>
    
    <!-- Modal Ver Detalle -->
    <div id="modalDetalle" class="modal">
        <div class="modal-content modal-lg">
            <span class="modal-close" onclick="cerrarModal()">&times;</span>
            <h2>Detalle de Compra</h2>
            <div id="detalleCompra"></div>
        </div>
    </div>
    
    <script>
        function verDetalleCompra(id) {
            fetch(`api/obtener_detalle_compra.php?id=${id}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        mostrarDetalle(data.compra, data.items);
                    } else {
                        alert('Error al cargar el detalle');
                    }
                });
        }
        
        function mostrarDetalle(compra, items) {
            let html = `
                <div class="compra-detalle">
                    <div class="detalle-header">
                        <p><strong>ID Compra:</strong> #${compra.id}</p>
                        <p><strong>Fecha:</strong> ${new Date(compra.fecha_compra).toLocaleString('es-PY')}</p>
                        <p><strong>Cliente:</strong> ${compra.nombre} ${compra.apellido}</p>
                        <p><strong>Cédula/RUC:</strong> ${compra.cedula_ruc}</p>
                        <p><strong>Estado:</strong> <span class="badge badge-${compra.estado === 'finalizada' ? 'success' : 'warning'}">${compra.estado}</span></p>
                    </div>
                    
                    <h3>Productos</h3>
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>Producto</th>
                                <th>Cantidad</th>
                                <th>Precio Unit.</th>
                                <th>Subtotal</th>
                            </tr>
                        </thead>
                        <tbody>
            `;
            
            items.forEach(item => {
                html += `
                    <tr>
                        <td>${item.nombre}</td>
                        <td>${item.cantidad}</td>
                        <td>${formatearGuaranies(item.precio_unitario)}</td>
                        <td><strong>${formatearGuaranies(item.subtotal)}</strong></td>
                    </tr>
                `;
            });
            
            html += `
                        </tbody>
                    </table>
                    
                    <div class="detalle-total">
                        <h3>TOTAL: ${formatearGuaranies(compra.total)}</h3>
                    </div>
                </div>
            `;
            
            document.getElementById('detalleCompra').innerHTML = html;
            document.getElementById('modalDetalle').style.display = 'block';
        }
        
        function cerrarModal() {
            document.getElementById('modalDetalle').style.display = 'none';
        }
    </script>
    <script src="../assets/js/admin.js"></script>
</body>
</html>
