<?php
/**
 * Gestión de clientes
 * Sistema de Supermercados y Tiendas V3.0
 */

require_once '../config/database.php';
require_once '../config/functions.php';

iniciarSesion();

if (!estaAutenticadoAdmin()) {
    redirigir('login.php');
}

if (!tienePermiso('gestionar_clientes')) {
    redirigir('dashboard.php');
}

$admin = obtenerAdminActual();
$db = getDB();

// Obtener todos los clientes
$stmt = $db->query("
    SELECT 
        c.*,
        COUNT(DISTINCT co.id) as total_compras,
        COALESCE(SUM(co.total), 0) as total_gastado
    FROM clientes c
    LEFT JOIN compras co ON c.id = co.cliente_id AND co.estado = 'finalizada'
    GROUP BY c.id
    ORDER BY c.fecha_registro DESC
");
$clientes = $stmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Clientes - Panel de Administración</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="../assets/css/admin.css">
</head>
<body class="admin-page">
    <?php include 'includes/header.php'; ?>
    <?php include 'includes/sidebar.php'; ?>
    
    <main class="admin-content">
        <div class="content-header">
            <h1>Gestión de Clientes</h1>
        </div>
        
        <!-- Estadísticas -->
        <div class="stats-grid">
            <div class="stat-card stat-primary">
                <div class="stat-icon">👥</div>
                <div class="stat-info">
                    <h3><?php echo count($clientes); ?></h3>
                    <p>Total de Clientes</p>
                </div>
            </div>
        </div>
        
        <!-- Filtros -->
        <div class="card">
            <div class="card-body">
                <input type="text" id="buscar" class="form-control" placeholder="Buscar por nombre, apellido o cédula..." onkeyup="filtrarClientes()">
            </div>
        </div>
        
        <!-- Tabla de clientes -->
        <div class="card">
            <div class="card-body">
                <div class="table-responsive">
                    <table class="data-table" id="tablaClientes">
                        <thead>
                            <tr>
                                <th>Cédula/RUC</th>
                                <th>Nombre Completo</th>
                                <th>Teléfono</th>
                                <th>Ciudad</th>
                                <th>Compras</th>
                                <th>Total Gastado</th>
                                <th>Fecha Registro</th>
                                <th>Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($clientes as $cliente): ?>
                                <tr data-nombre="<?php echo htmlspecialchars(strtolower($cliente['nombre'] . ' ' . $cliente['apellido'])); ?>"
                                    data-cedula="<?php echo htmlspecialchars($cliente['cedula_ruc']); ?>">
                                    <td><?php echo htmlspecialchars(formatearCedula($cliente['cedula_ruc'])); ?></td>
                                    <td>
                                        <strong><?php echo htmlspecialchars($cliente['nombre'] . ' ' . $cliente['apellido']); ?></strong>
                                    </td>
                                    <td><?php echo htmlspecialchars($cliente['telefono'] ?? 'N/A'); ?></td>
                                    <td><?php echo htmlspecialchars($cliente['ciudad'] ?? 'N/A'); ?></td>
                                    <td><?php echo $cliente['total_compras']; ?></td>
                                    <td><?php echo formatearGuaranies($cliente['total_gastado']); ?></td>
                                    <td><?php echo date('d/m/Y', strtotime($cliente['fecha_registro'])); ?></td>
                                    <td class="actions-cell">
                                        <button class="btn btn-sm btn-info" onclick='verCliente(<?php echo json_encode($cliente); ?>)'>
                                            👁️
                                        </button>
                                        <button class="btn btn-sm btn-warning" onclick='editarCliente(<?php echo json_encode($cliente); ?>)'>
                                            ✏️
                                        </button>
                                        <?php if (tienePermiso('eliminar')): ?>
                                            <button class="btn btn-sm btn-danger" onclick="eliminarCliente(<?php echo $cliente['id']; ?>)">
                                                🗑️
                                            </button>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </main>
    
    <!-- Modal Ver Cliente -->
    <div id="modalVerCliente" class="modal">
        <div class="modal-content">
            <span class="modal-close" onclick="cerrarModalVer()">&times;</span>
            <h2>Información del Cliente</h2>
            <div id="infoCliente"></div>
        </div>
    </div>
    
    <!-- Modal Editar Cliente -->
    <div id="modalEditarCliente" class="modal">
        <div class="modal-content">
            <span class="modal-close" onclick="cerrarModalEditar()">&times;</span>
            <h2>Editar Cliente</h2>
            <form id="formEditarCliente" onsubmit="guardarCliente(event)">
                <input type="hidden" id="cliente_id" name="id">
                
                <div class="form-group">
                    <label>Cédula/RUC</label>
                    <input type="text" id="cedula_ruc" class="form-control" disabled>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="nombre">Nombre *</label>
                        <input type="text" id="nombre" name="nombre" class="form-control" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="apellido">Apellido *</label>
                        <input type="text" id="apellido" name="apellido" class="form-control" required>
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="telefono">Teléfono</label>
                        <input type="tel" id="telefono" name="telefono" class="form-control">
                    </div>
                    
                    <div class="form-group">
                        <label for="ciudad">Ciudad</label>
                        <input type="text" id="ciudad" name="ciudad" class="form-control">
                    </div>
                </div>
                
                <div class="form-actions">
                    <button type="button" class="btn btn-secondary" onclick="cerrarModalEditar()">Cancelar</button>
                    <button type="submit" class="btn btn-primary">Guardar Cambios</button>
                </div>
            </form>
        </div>
    </div>
    
    <script>
        function filtrarClientes() {
            const buscar = document.getElementById('buscar').value.toLowerCase();
            const filas = document.querySelectorAll('#tablaClientes tbody tr');
            
            filas.forEach(fila => {
                const nombre = fila.dataset.nombre;
                const cedula = fila.dataset.cedula;
                
                if (nombre.includes(buscar) || cedula.includes(buscar)) {
                    fila.style.display = '';
                } else {
                    fila.style.display = 'none';
                }
            });
        }
        
        function verCliente(cliente) {
            const html = `
                <div class="cliente-detalle">
                    <p><strong>Cédula/RUC:</strong> ${cliente.cedula_ruc}</p>
                    <p><strong>Nombre:</strong> ${cliente.nombre} ${cliente.apellido}</p>
                    <p><strong>Teléfono:</strong> ${cliente.telefono || 'N/A'}</p>
                    <p><strong>Ciudad:</strong> ${cliente.ciudad || 'N/A'}</p>
                    <p><strong>Total de Compras:</strong> ${cliente.total_compras}</p>
                    <p><strong>Total Gastado:</strong> ${formatearGuaranies(cliente.total_gastado)}</p>
                    <p><strong>Fecha de Registro:</strong> ${new Date(cliente.fecha_registro).toLocaleDateString('es-PY')}</p>
                </div>
            `;
            document.getElementById('infoCliente').innerHTML = html;
            document.getElementById('modalVerCliente').style.display = 'block';
        }
        
        function cerrarModalVer() {
            document.getElementById('modalVerCliente').style.display = 'none';
        }
        
        function editarCliente(cliente) {
            document.getElementById('cliente_id').value = cliente.id;
            document.getElementById('cedula_ruc').value = cliente.cedula_ruc;
            document.getElementById('nombre').value = cliente.nombre;
            document.getElementById('apellido').value = cliente.apellido;
            document.getElementById('telefono').value = cliente.telefono || '';
            document.getElementById('ciudad').value = cliente.ciudad || '';
            document.getElementById('modalEditarCliente').style.display = 'block';
        }
        
        function cerrarModalEditar() {
            document.getElementById('modalEditarCliente').style.display = 'none';
        }
        
        function guardarCliente(event) {
            event.preventDefault();
            
            const formData = new FormData(document.getElementById('formEditarCliente'));
            const data = Object.fromEntries(formData);
            
            fetch('api/guardar_cliente.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(data)
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert('Cliente actualizado exitosamente');
                    location.reload();
                } else {
                    alert('Error: ' + (data.message || 'No se pudo actualizar el cliente'));
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Error al actualizar el cliente');
            });
        }
        
        function eliminarCliente(id) {
            if (confirm('¿Está seguro de eliminar este cliente? Esta acción no se puede deshacer.')) {
                fetch('api/eliminar_cliente.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ id: id })
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        alert('Cliente eliminado exitosamente');
                        location.reload();
                    } else {
                        alert('Error: ' + (data.message || 'No se pudo eliminar el cliente'));
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Error al eliminar el cliente');
                });
            }
        }
    </script>
    <script src="../assets/js/admin.js"></script>
</body>
</html>
