<?php
/**
 * Nueva Venta - Vendedor
 * Sistema de Gestión de Agua Mineral
 */

require_once '../../includes/init.php';

// Verificar que sea vendedor
if ($_SESSION['user_role'] !== 'vendedor') {
    redirect('../admin/dashboard.php');
}

$pageTitle = 'Nueva Venta - Vendedor';

$db = Database::getInstance();
$vendedorId = $_SESSION['user_id'];

// Obtener cliente preseleccionado si viene de la URL
$clientePreseleccionado = $_GET['cliente_id'] ?? '';

// Procesar formulario de venta
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $clienteId = (int)$_POST['cliente_id'];
    $fecha = $_POST['fecha'];
    $cantidadBidones = (int)$_POST['cantidad_bidones'];
    $observaciones = sanitize($_POST['observaciones'] ?? '');
    
    // Validaciones
    $errores = [];
    
    if (empty($clienteId)) {
        $errores[] = 'Debe seleccionar un cliente.';
    }
    
    if (empty($fecha)) {
        $errores[] = 'La fecha es requerida.';
    }
    
    if ($cantidadBidones <= 0) {
        $errores[] = 'La cantidad de bidones debe ser mayor a 0.';
    }
    
    // Verificar que el cliente pertenezca al vendedor
    if (!empty($clienteId)) {
        $clienteCheck = $db->select("SELECT id FROM clientes WHERE id = ? AND vendedor_id = ? AND activo = 1", [$clienteId, $vendedorId]);
        if (empty($clienteCheck)) {
            $errores[] = 'Cliente no válido o no pertenece a su zona.';
        }
    }
    
    if (empty($errores)) {
        // Obtener precio actual del producto
        $producto = $db->select("SELECT precio FROM productos WHERE activo = 1 ORDER BY id LIMIT 1")[0];
        $precioUnitario = $producto['precio'];
        
        // Insertar la venta
        $query = "INSERT INTO ventas (cliente_id, fecha, cantidad_bidones, precio_unitario, observaciones) VALUES (?, ?, ?, ?, ?)";
        
        if ($db->execute($query, [$clienteId, $fecha, $cantidadBidones, $precioUnitario, $observaciones])) {
            $ventaId = $db->lastInsertId();
            
            // Log de actividad
            logActivity($_SESSION['user_id'], 'create_sale', "Venta creada: ID $ventaId, Cliente $clienteId, $cantidadBidones bidones");
            
            setFlashMessage('Venta registrada correctamente.', 'success');
            redirect('dashboard.php');
        } else {
            $errores[] = 'Error al registrar la venta. Intente nuevamente.';
        }
    }
}

// Obtener clientes del vendedor
$clientes = $db->select("
    SELECT 
        c.id, 
        c.nombre, 
        c.direccion,
        (SELECT COUNT(*) FROM ventas v WHERE v.cliente_id = c.id AND MONTH(v.fecha) = MONTH(CURDATE()) AND YEAR(v.fecha) = YEAR(CURDATE())) as entregas_mes
    FROM clientes c 
    WHERE c.vendedor_id = ? AND c.activo = 1 
    ORDER BY c.nombre
", [$vendedorId]);

// Obtener precio actual
$producto = $db->select("SELECT * FROM productos WHERE activo = 1 ORDER BY id LIMIT 1")[0];

include '../../includes/header.php';
?>

<div class="row">
    <div class="col-12">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h1 class="h3">
                <i class="fas fa-plus-circle me-2"></i>Nueva Venta
            </h1>
            <a href="dashboard.php" class="btn btn-secondary">
                <i class="fas fa-arrow-left me-2"></i>Volver al Dashboard
            </a>
        </div>
    </div>
</div>

<?php if (!empty($errores)): ?>
    <div class="alert alert-danger">
        <ul class="mb-0">
            <?php foreach ($errores as $error): ?>
                <li><?= $error ?></li>
            <?php endforeach; ?>
        </ul>
    </div>
<?php endif; ?>

<div class="row">
    <div class="col-lg-8">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Registrar Nueva Venta</h5>
            </div>
            <div class="card-body">
                <form method="POST" id="ventaForm">
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="cliente_id" class="form-label">Cliente *</label>
                            <select name="cliente_id" id="cliente_id" class="form-select" required>
                                <option value="">Seleccionar cliente...</option>
                                <?php foreach ($clientes as $cliente): ?>
                                    <option value="<?= $cliente['id'] ?>" 
                                            <?= $clientePreseleccionado == $cliente['id'] ? 'selected' : '' ?>
                                            data-direccion="<?= htmlspecialchars($cliente['direccion']) ?>"
                                            data-entregas="<?= $cliente['entregas_mes'] ?>">
                                        <?= htmlspecialchars($cliente['nombre']) ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <div id="clienteInfo" class="mt-2" style="display: none;">
                                <small class="text-muted">
                                    <i class="fas fa-map-marker-alt me-1"></i>
                                    <span id="clienteDireccion"></span>
                                </small>
                                <br>
                                <small class="text-info">
                                    <i class="fas fa-truck me-1"></i>
                                    <span id="clienteEntregas"></span> entregas este mes
                                </small>
                            </div>
                        </div>
                        
                        <div class="col-md-6 mb-3">
                            <label for="fecha" class="form-label">Fecha de Entrega *</label>
                            <input type="date" name="fecha" id="fecha" class="form-control" 
                                   value="<?= date('Y-m-d') ?>" max="<?= date('Y-m-d') ?>" required>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="cantidad_bidones" class="form-label">Cantidad de Bidones *</label>
                            <div class="input-group">
                                <button type="button" class="btn btn-outline-secondary" onclick="cambiarCantidad(-1)">
                                    <i class="fas fa-minus"></i>
                                </button>
                                <input type="number" name="cantidad_bidones" id="cantidad_bidones" 
                                       class="form-control text-center" value="1" min="1" max="50" required>
                                <button type="button" class="btn btn-outline-secondary" onclick="cambiarCantidad(1)">
                                    <i class="fas fa-plus"></i>
                                </button>
                            </div>
                        </div>
                        
                        <div class="col-md-6 mb-3">
                            <label for="precio_unitario" class="form-label">Precio Unitario</label>
                            <div class="input-group">
                                <span class="input-group-text">$</span>
                                <input type="text" id="precio_unitario" class="form-control" 
                                       value="<?= formatCurrency($producto['precio']) ?>" readonly>
                            </div>
                            <small class="text-muted">Precio actual del sistema</small>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="observaciones" class="form-label">Observaciones</label>
                        <textarea name="observaciones" id="observaciones" class="form-control" rows="3" 
                                  placeholder="Observaciones adicionales sobre la entrega (opcional)"></textarea>
                    </div>
                    
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <strong>Total: <span id="totalVenta" class="text-primary h4"><?= formatCurrency($producto['precio']) ?></span></strong>
                        </div>
                        <div>
                            <button type="submit" class="btn btn-primary btn-lg">
                                <i class="fas fa-save me-2"></i>Registrar Venta
                            </button>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <div class="col-lg-4">
        <!-- Información del producto -->
        <div class="card mb-4">
            <div class="card-header">
                <h6 class="mb-0">Información del Producto</h6>
            </div>
            <div class="card-body">
                <div class="text-center">
                    <i class="fas fa-tint fa-3x text-primary mb-3"></i>
                    <h5><?= htmlspecialchars($producto['nombre']) ?></h5>
                    <div class="h4 text-success"><?= formatCurrency($producto['precio']) ?></div>
                    <small class="text-muted">Precio por unidad</small>
                </div>
            </div>
        </div>
        
        <!-- Ventas recientes -->
        <div class="card">
            <div class="card-header">
                <h6 class="mb-0">Mis Últimas Ventas</h6>
            </div>
            <div class="card-body">
                <?php
                $ventasRecientes = $db->select("
                    SELECT 
                        c.nombre as cliente_nombre,
                        v.fecha,
                        v.cantidad_bidones,
                        v.total
                    FROM ventas v
                    INNER JOIN clientes c ON v.cliente_id = c.id
                    WHERE c.vendedor_id = ?
                    ORDER BY v.created_at DESC
                    LIMIT 5
                ", [$vendedorId]);
                ?>
                
                <?php if (empty($ventasRecientes)): ?>
                    <p class="text-muted text-center">No hay ventas registradas</p>
                <?php else: ?>
                    <?php foreach ($ventasRecientes as $venta): ?>
                        <div class="d-flex justify-content-between align-items-center mb-2 p-2 border rounded">
                            <div>
                                <div class="fw-bold"><?= htmlspecialchars($venta['cliente_nombre']) ?></div>
                                <small class="text-muted"><?= formatDate($venta['fecha']) ?></small>
                            </div>
                            <div class="text-end">
                                <div class="fw-bold"><?= $venta['cantidad_bidones'] ?> bidones</div>
                                <small class="text-success"><?= formatCurrency($venta['total']) ?></small>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<?php
$extraJS = '
<script>
const precioUnitario = ' . $producto['precio'] . ';

// Mostrar información del cliente seleccionado
document.getElementById("cliente_id").addEventListener("change", function() {
    const option = this.options[this.selectedIndex];
    const clienteInfo = document.getElementById("clienteInfo");
    
    if (this.value) {
        document.getElementById("clienteDireccion").textContent = option.dataset.direccion;
        document.getElementById("clienteEntregas").textContent = option.dataset.entregas;
        clienteInfo.style.display = "block";
    } else {
        clienteInfo.style.display = "none";
    }
});

// Cambiar cantidad con botones
function cambiarCantidad(delta) {
    const input = document.getElementById("cantidad_bidones");
    let valor = parseInt(input.value) + delta;
    
    if (valor < 1) valor = 1;
    if (valor > 50) valor = 50;
    
    input.value = valor;
    actualizarTotal();
}

// Actualizar total cuando cambia la cantidad
document.getElementById("cantidad_bidones").addEventListener("input", actualizarTotal);

function actualizarTotal() {
    const cantidad = parseInt(document.getElementById("cantidad_bidones").value) || 1;
    const total = cantidad * precioUnitario;
    
    document.getElementById("totalVenta").textContent = "$" + total.toLocaleString("es-AR", {
        minimumFractionDigits: 2,
        maximumFractionDigits: 2
    });
}

// Validación del formulario
document.getElementById("ventaForm").addEventListener("submit", function(e) {
    const clienteId = document.getElementById("cliente_id").value;
    const cantidad = parseInt(document.getElementById("cantidad_bidones").value);
    
    if (!clienteId) {
        e.preventDefault();
        alert("Por favor, seleccione un cliente.");
        return;
    }
    
    if (cantidad <= 0) {
        e.preventDefault();
        alert("La cantidad debe ser mayor a 0.");
        return;
    }
    
    // Confirmación
    const cliente = document.getElementById("cliente_id").options[document.getElementById("cliente_id").selectedIndex].text;
    const total = document.getElementById("totalVenta").textContent;
    
    if (!confirm(`¿Confirma la venta de ${cantidad} bidones a ${cliente} por ${total}?`)) {
        e.preventDefault();
    }
});

// Inicializar información si hay cliente preseleccionado
if (document.getElementById("cliente_id").value) {
    document.getElementById("cliente_id").dispatchEvent(new Event("change"));
}
</script>
';

include '../../includes/footer.php';
?>
