<?php
/**
 * Gestión de Vendedores - Administrador
 * Sistema de Gestión de Agua Mineral
 */

require_once '../../includes/init.php';

// Verificar que sea administrador
checkPermission('administrador');

$pageTitle = 'Gestión de Vendedores - Administrador';

$db = Database::getInstance();

// Procesar acciones
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'create') {
        $nombre = sanitize($_POST['nombre']);
        $usuario = sanitize($_POST['usuario']);
        $password = $_POST['password'];
        $zonaId = (int)$_POST['zona_id'];
        $telefono = sanitize($_POST['telefono']);
        $email = sanitize($_POST['email']);
        
        // Validaciones
        $errors = [];
        
        if (empty($nombre)) $errors[] = 'El nombre es obligatorio.';
        if (empty($usuario)) $errors[] = 'El usuario es obligatorio.';
        if (empty($password)) $errors[] = 'La contraseña es obligatoria.';
        if (strlen($password) < 6) $errors[] = 'La contraseña debe tener al menos 6 caracteres.';
        
        // Verificar que el usuario no exista
        if ($usuario) {
            $existing = $db->select("SELECT id FROM usuarios WHERE usuario = ?", [$usuario]);
            if (!empty($existing)) {
                $errors[] = 'El nombre de usuario ya existe.';
            }
        }
        
        if (empty($errors)) {
            $hashedPassword = hashPassword($password);
            $query = "INSERT INTO usuarios (nombre, usuario, password, rol, zona_id, telefono, email, fecha_registro) 
                      VALUES (?, ?, ?, 'vendedor', ?, ?, ?, NOW())";
            
            if ($db->execute($query, [$nombre, $usuario, $hashedPassword, $zonaId, $telefono, $email])) {
                logActivity($_SESSION['user_id'], 'create_seller', "Vendedor creado: $nombre");
                setFlashMessage('Vendedor creado exitosamente.', 'success');
                redirect('vendedores.php');
            } else {
                setFlashMessage('Error al crear el vendedor.', 'error');
            }
        } else {
            foreach ($errors as $error) {
                setFlashMessage($error, 'error');
            }
        }
    }
    
    if ($action === 'toggle_status') {
        $vendedorId = (int)$_POST['vendedor_id'];
        $newStatus = (int)$_POST['new_status'];
        
        $query = "UPDATE usuarios SET activo = ? WHERE id = ? AND rol = 'vendedor'";
        if ($db->execute($query, [$newStatus, $vendedorId])) {
            $status = $newStatus ? 'activado' : 'desactivado';
            logActivity($_SESSION['user_id'], 'toggle_seller_status', "Vendedor $status: ID $vendedorId");
            setFlashMessage("Vendedor $status exitosamente.", 'success');
        } else {
            setFlashMessage('Error al cambiar el estado del vendedor.', 'error');
        }
        redirect('vendedores.php');
    }
    
    if ($action === 'delete_seller') {
        $vendedorId = (int)$_POST['vendedor_id'];
        
        // Verificar si tiene clientes asignados
        $hasClients = $db->select("SELECT COUNT(*) as count FROM clientes WHERE vendedor_id = ?", [$vendedorId])[0]['count'];
        
        if ($hasClients > 0) {
            setFlashMessage('No se puede eliminar el vendedor porque tiene clientes asignados. Desactívelo en su lugar.', 'error');
        } else {
            if ($db->execute("DELETE FROM usuarios WHERE id = ? AND rol = 'vendedor'", [$vendedorId])) {
                logActivity($_SESSION['user_id'], 'delete_seller', "Vendedor eliminado permanentemente: ID $vendedorId");
                setFlashMessage('Vendedor eliminado permanentemente.', 'success');
            } else {
                setFlashMessage('Error al eliminar el vendedor.', 'error');
            }
        }
        redirect('vendedores.php');
    }
}

// Obtener vendedores con estadísticas
$vendedores = $db->select("
    SELECT 
        u.id,
        u.nombre,
        u.usuario,
        u.telefono,
        u.email,
        u.activo,
        u.fecha_registro,
        z.nombre as zona_nombre,
        COUNT(DISTINCT c.id) as total_clientes,
        COUNT(DISTINCT CASE WHEN c.activo = 1 THEN c.id END) as clientes_activos,
        COALESCE(SUM(v.cantidad_bidones), 0) as total_bidones_vendidos,
        COALESCE(SUM(v.total), 0) as total_facturado,
        COALESCE(SUM(CASE WHEN MONTH(v.fecha) = MONTH(CURDATE()) AND YEAR(v.fecha) = YEAR(CURDATE()) THEN v.total ELSE 0 END), 0) as facturado_mes
    FROM usuarios u
    LEFT JOIN zonas z ON u.zona_id = z.id
    LEFT JOIN clientes c ON u.id = c.vendedor_id
    LEFT JOIN ventas v ON c.id = v.cliente_id
    WHERE u.rol = 'vendedor'
    GROUP BY u.id, u.nombre, u.usuario, u.telefono, u.email, u.activo, u.fecha_registro, z.nombre
    ORDER BY u.activo DESC, u.nombre ASC
");

// Obtener zonas para el formulario
$zonas = $db->select("SELECT id, nombre FROM zonas ORDER BY nombre");

include '../../includes/header.php';
?>

<div class="row">
    <div class="col-12">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h1 class="h3">
                <i class="fas fa-user-tie me-2"></i>Gestión de Vendedores
            </h1>
            <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#nuevoVendedorModal">
                <i class="fas fa-plus me-2"></i>Nuevo Vendedor
            </button>
        </div>
    </div>
</div>

<!-- Estadísticas generales -->
<div class="row mb-4">
    <?php
    $statsGenerales = $db->select("
        SELECT 
            COUNT(*) as total_vendedores,
            COUNT(CASE WHEN activo = 1 THEN 1 END) as vendedores_activos,
            COUNT(CASE WHEN activo = 0 THEN 1 END) as vendedores_inactivos
        FROM usuarios WHERE rol = 'vendedor'
    ")[0];
    ?>
    
    <div class="col-md-4">
        <div class="card bg-primary text-white">
            <div class="card-body text-center">
                <i class="fas fa-users fa-2x mb-2"></i>
                <h4><?= $statsGenerales['total_vendedores'] ?></h4>
                <small>Total Vendedores</small>
            </div>
        </div>
    </div>
    
    <div class="col-md-4">
        <div class="card bg-success text-white">
            <div class="card-body text-center">
                <i class="fas fa-user-check fa-2x mb-2"></i>
                <h4><?= $statsGenerales['vendedores_activos'] ?></h4>
                <small>Activos</small>
            </div>
        </div>
    </div>
    
    <div class="col-md-4">
        <div class="card bg-secondary text-white">
            <div class="card-body text-center">
                <i class="fas fa-user-times fa-2x mb-2"></i>
                <h4><?= $statsGenerales['vendedores_inactivos'] ?></h4>
                <small>Inactivos</small>
            </div>
        </div>
    </div>
</div>

<!-- Lista de vendedores -->
<div class="card">
    <div class="card-header">
        <h5 class="mb-0">Lista de Vendedores</h5>
    </div>
    <div class="card-body">
        <div class="table-responsive">
            <table class="table table-striped table-hover" id="vendedoresTable">
                <thead class="table-dark">
                    <tr>
                        <th>Vendedor</th>
                        <th>Usuario</th>
                        <th>Zona</th>
                        <th>Contacto</th>
                        <th>Clientes</th>
                        <th>Ventas</th>
                        <th>Facturado</th>
                        <th>Estado</th>
                        <th>Acciones</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($vendedores as $vendedor): ?>
                        <tr class="<?= $vendedor['activo'] ? '' : 'table-secondary' ?>">
                            <td>
                                <div class="d-flex align-items-center">
                                    <div class="avatar-sm bg-<?= $vendedor['activo'] ? 'success' : 'secondary' ?> rounded-circle d-flex align-items-center justify-content-center me-2">
                                        <i class="fas fa-user text-white"></i>
                                    </div>
                                    <div>
                                        <div class="fw-bold"><?= htmlspecialchars($vendedor['nombre']) ?></div>
                                        <small class="text-muted">Desde <?= formatDate($vendedor['fecha_registro']) ?></small>
                                    </div>
                                </div>
                            </td>
                            <td>
                                <span class="badge bg-info"><?= htmlspecialchars($vendedor['usuario']) ?></span>
                            </td>
                            <td>
                                <?= htmlspecialchars($vendedor['zona_nombre'] ?? 'Sin asignar') ?>
                            </td>
                            <td>
                                <?php if ($vendedor['telefono']): ?>
                                    <div><i class="fas fa-phone me-1"></i><?= htmlspecialchars($vendedor['telefono']) ?></div>
                                <?php endif; ?>
                                <?php if ($vendedor['email']): ?>
                                    <div><i class="fas fa-envelope me-1"></i><?= htmlspecialchars($vendedor['email']) ?></div>
                                <?php endif; ?>
                            </td>
                            <td>
                                <span class="badge bg-primary"><?= $vendedor['clientes_activos'] ?></span>
                                <small class="text-muted">/ <?= $vendedor['total_clientes'] ?></small>
                            </td>
                            <td>
                                <div><?= safeNumber($vendedor['total_bidones_vendidos']) ?> bidones</div>
                                <small class="text-success"><?= safeCurrency($vendedor['facturado_mes']) ?> este mes</small>
                            </td>
                            <td>
                                <div class="fw-bold"><?= safeCurrency($vendedor['total_facturado']) ?></div>
                            </td>
                            <td>
                                <span class="badge bg-<?= $vendedor['activo'] ? 'success' : 'secondary' ?>">
                                    <?= $vendedor['activo'] ? 'Activo' : 'Inactivo' ?>
                                </span>
                            </td>
                            <td>
                                <div class="btn-group btn-group-sm">
                                    <a href="vendedor-editar.php?id=<?= $vendedor['id'] ?>" 
                                       class="btn btn-outline-primary" title="Editar">
                                        <i class="fas fa-edit"></i>
                                    </a>                                    
                                    <a href="vendedor-editar.php?id=<?= $vendedor['id'] ?>" 
                                       class="btn btn-outline-primary" title="Editar">
                                        <i class="fas fa-edit"></i>
                                    </a>
                                    
                                    <button type="button" class="btn btn-outline-danger" 
                                            onclick="deleteSeller(<?= $vendedor['id'] ?>, '<?= htmlspecialchars($vendedor['nombre']) ?>')"
                                            title="Eliminar permanentemente">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                    
                                    <button type="button" class="btn btn-outline-<?= $vendedor['activo'] ? 'warning' : 'success' ?>" 
                                            onclick="toggleSellerStatus(<?= $vendedor['id'] ?>, <?= $vendedor['activo'] ? 0 : 1 ?>)"
                                            title="<?= $vendedor['activo'] ? 'Desactivar' : 'Activar' ?>">
                                        <i class="fas fa-<?= $vendedor['activo'] ? 'pause' : 'play' ?>"></i>
                                    </button>
                                </div>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Modal Nuevo Vendedor -->
<div class="modal fade" id="nuevoVendedorModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Nuevo Vendedor</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST" class="needs-validation" novalidate>
                <input type="hidden" name="action" value="create">
                <div class="modal-body">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="nombre" class="form-label">Nombre Completo *</label>
                                <input type="text" name="nombre" id="nombre" class="form-control" required>
                                <div class="invalid-feedback">
                                    Por favor ingrese el nombre completo.
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="usuario" class="form-label">Usuario *</label>
                                <input type="text" name="usuario" id="usuario" class="form-control" required>
                                <div class="invalid-feedback">
                                    Por favor ingrese un nombre de usuario.
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="password" class="form-label">Contraseña *</label>
                                <input type="password" name="password" id="password" class="form-control" 
                                       minlength="6" required>
                                <div class="invalid-feedback">
                                    La contraseña debe tener al menos 6 caracteres.
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="zona_id" class="form-label">Zona</label>
                                <select name="zona_id" id="zona_id" class="form-select">
                                    <option value="">Sin asignar</option>
                                    <?php foreach ($zonas as $zona): ?>
                                        <option value="<?= $zona['id'] ?>"><?= htmlspecialchars($zona['nombre']) ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="telefono" class="form-label">Teléfono</label>
                                <input type="tel" name="telefono" id="telefono" class="form-control" 
                                       placeholder="0981-123-456">
                            </div>
                        </div>
                        
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="email" class="form-label">Email</label>
                                <input type="email" name="email" id="email" class="form-control">
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-primary">Crear Vendedor</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Modal de confirmación de estado -->
<div class="modal fade" id="confirmStatusModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Confirmar Acción</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <p id="confirmMessage"></p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                <form method="POST" style="display: inline;">
                    <input type="hidden" name="action" value="toggle_status">
                    <input type="hidden" name="vendedor_id" id="confirmVendedorId">
                    <input type="hidden" name="new_status" id="confirmNewStatus">
                    <button type="submit" class="btn btn-primary">Confirmar</button>
                </form>
            </div>
        </div>
    </div>
</div>

<!-- Modal de confirmación de eliminación -->
<div class="modal fade" id="confirmDeleteModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title text-danger">Eliminar Vendedor</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <p class="text-danger fw-bold">ADVERTENCIA: Esta acción es irreversible.</p>
                <p>¿Está seguro de que desea eliminar permanentemente al vendedor <strong id="deleteSellerName"></strong>?</p>
                <p class="small text-muted">Solo se podrá eliminar si no tiene clientes asignados.</p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                <form method="POST" style="display: inline;">
                    <input type="hidden" name="action" value="delete_seller">
                    <input type="hidden" name="vendedor_id" id="deleteSellerId">
                    <button type="submit" class="btn btn-danger">Eliminar Permanentemente</button>
                </form>
            </div>
        </div>
    </div>
</div>
<div class="modal fade" id="nuevoVendedorModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Nuevo Vendedor</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST" class="needs-validation" novalidate>
                <input type="hidden" name="action" value="create">
                <div class="modal-body">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="nombre" class="form-label">Nombre Completo *</label>
                                <input type="text" name="nombre" id="nombre" class="form-control" required>
                                <div class="invalid-feedback">
                                    Por favor ingrese el nombre completo.
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="usuario" class="form-label">Usuario *</label>
                                <input type="text" name="usuario" id="usuario" class="form-control" required>
                                <div class="invalid-feedback">
                                    Por favor ingrese un nombre de usuario.
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="password" class="form-label">Contraseña *</label>
                                <input type="password" name="password" id="password" class="form-control" 
                                       minlength="6" required>
                                <div class="invalid-feedback">
                                    La contraseña debe tener al menos 6 caracteres.
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="zona_id" class="form-label">Zona</label>
                                <select name="zona_id" id="zona_id" class="form-select">
                                    <option value="">Sin asignar</option>
                                    <?php foreach ($zonas as $zona): ?>
                                        <option value="<?= $zona['id'] ?>"><?= htmlspecialchars($zona['nombre']) ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="telefono" class="form-label">Teléfono</label>
                                <input type="tel" name="telefono" id="telefono" class="form-control" 
                                       placeholder="0981-123-456">
                            </div>
                        </div>
                        
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="email" class="form-label">Email</label>
                                <input type="email" name="email" id="email" class="form-control">
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-primary">Crear Vendedor</button>
                </div>
            </form>
        </div>
    </div>
</div>

<?php
$extraJS = '
<script>
// DataTable
$(document).ready(function() {
    $("#vendedoresTable").DataTable({
        "language": {
            "url": "//cdn.datatables.net/plug-ins/1.10.24/i18n/Spanish.json"
        },
        "pageLength": 25,
        "order": [[7, "desc"], [0, "asc"]], // Ordenar por estado y nombre
        "columnDefs": [
            { "orderable": false, "targets": 8 } // Columna de acciones no ordenable
        ]
    });
});

// Validación del formulario
(function() {
    "use strict";
    window.addEventListener("load", function() {
        var forms = document.getElementsByClassName("needs-validation");
        var validation = Array.prototype.filter.call(forms, function(form) {
            form.addEventListener("submit", function(event) {
                if (form.checkValidity() === false) {
                    event.preventDefault();
                    event.stopPropagation();
                }
                form.classList.add("was-validated");
            }, false);
        });
    }, false);
})();

// Funciones para acciones
function verDetalles(vendedorId) {
    // Implementar vista de detalles
    alert("Función de detalles en desarrollo. Vendedor ID: " + vendedorId);
}

function editarVendedor(vendedorId) {
    // Redirigir a la página de edición
    window.location.href = 'vendedor-editar.php?id=' + vendedorId;
}

function toggleSellerStatus(vendedorId, newStatus) {
    const action = newStatus ? "activar" : "desactivar";
    document.getElementById("confirmMessage").textContent = 
        `¿Está seguro que desea ${action} este vendedor?`;
    document.getElementById("confirmVendedorId").value = vendedorId;
    document.getElementById("confirmNewStatus").value = newStatus;
    
    const modal = new bootstrap.Modal(document.getElementById("confirmStatusModal"));
    modal.show();
}

function deleteSeller(vendedorId, vendedorNombre) {
    document.getElementById("deleteSellerId").value = vendedorId;
    document.getElementById("deleteSellerName").textContent = vendedorNombre;
    
    const modal = new bootstrap.Modal(document.getElementById("confirmDeleteModal"));
    modal.show();
}
</script>

<link rel="stylesheet" type="text/css" href="https://cdn.datatables.net/1.10.24/css/dataTables.bootstrap5.min.css">
<script type="text/javascript" charset="utf8" src="https://cdn.datatables.net/1.10.24/js/jquery.dataTables.js"></script>
<script type="text/javascript" charset="utf8" src="https://cdn.datatables.net/1.10.24/js/dataTables.bootstrap5.min.js"></script>
';

include '../../includes/footer.php';
?>
