<?php
/**
 * Reportes y Exportación - Administrador
 * Sistema de Gestión de Agua Mineral
 */

require_once '../../includes/init.php';

// Verificar que sea administrador
checkPermission('administrador');

$pageTitle = 'Reportes y Exportación - Administrador';

$db = Database::getInstance();

// Parámetros de filtro
$fechaInicio = $_GET['fecha_inicio'] ?? date('Y-m-01'); // Primer día del mes actual
$fechaFin = $_GET['fecha_fin'] ?? date('Y-m-t'); // Último día del mes actual
$vendedorId = $_GET['vendedor_id'] ?? '';
$clienteId = $_GET['cliente_id'] ?? '';
$tipoReporte = $_GET['tipo_reporte'] ?? 'ventas_detalle';

// Procesar exportación
if (isset($_GET['export'])) {
    $exportType = $_GET['export'];
    
    // Generar datos según el tipo de reporte
    $datos = obtenerDatosReporte($db, $tipoReporte, $fechaInicio, $fechaFin, $vendedorId, $clienteId);
    
    if ($exportType === 'excel') {
        exportarExcel($datos, $tipoReporte, $fechaInicio, $fechaFin);
    } elseif ($exportType === 'pdf') {
        exportarPDF($datos, $tipoReporte, $fechaInicio, $fechaFin);
    }
    exit;
}

// Obtener datos para los filtros
$vendedores = $db->select("SELECT id, nombre FROM usuarios WHERE rol = 'vendedor' AND activo = 1 ORDER BY nombre");
$clientes = $db->select("
    SELECT c.id, c.nombre, u.nombre as vendedor_nombre 
    FROM clientes c 
    INNER JOIN usuarios u ON c.vendedor_id = u.id 
    WHERE c.activo = 1 
    ORDER BY c.nombre
");

// Obtener datos del reporte actual
$datosReporte = obtenerDatosReporte($db, $tipoReporte, $fechaInicio, $fechaFin, $vendedorId, $clienteId);

// Función para obtener datos del reporte
function obtenerDatosReporte($db, $tipo, $fechaInicio, $fechaFin, $vendedorId = '', $clienteId = '') {
    $whereConditions = ["v.fecha BETWEEN ? AND ?"];
    $params = [$fechaInicio, $fechaFin];
    
    if (!empty($vendedorId)) {
        $whereConditions[] = "c.vendedor_id = ?";
        $params[] = $vendedorId;
    }
    
    if (!empty($clienteId)) {
        $whereConditions[] = "c.id = ?";
        $params[] = $clienteId;
    }
    
    $whereClause = 'WHERE ' . implode(' AND ', $whereConditions);
    
    switch ($tipo) {
        case 'ventas_detalle':
            return $db->select("
                SELECT 
                    v.fecha,
                    c.nombre as cliente,
                    u.nombre as vendedor,
                    v.cantidad_bidones,
                    v.precio_unitario,
                    v.total,
                    v.observaciones
                FROM ventas v
                INNER JOIN clientes c ON v.cliente_id = c.id
                INNER JOIN usuarios u ON c.vendedor_id = u.id
                $whereClause
                ORDER BY v.fecha DESC, c.nombre
            ", $params);
            
        case 'ventas_por_cliente':
            return $db->select("
                SELECT 
                    c.nombre as cliente,
                    u.nombre as vendedor,
                    COUNT(v.id) as total_entregas,
                    SUM(v.cantidad_bidones) as total_bidones,
                    SUM(v.total) as total_ventas,
                    AVG(v.cantidad_bidones) as promedio_bidones,
                    MIN(v.fecha) as primera_venta,
                    MAX(v.fecha) as ultima_venta
                FROM ventas v
                INNER JOIN clientes c ON v.cliente_id = c.id
                INNER JOIN usuarios u ON c.vendedor_id = u.id
                $whereClause
                GROUP BY c.id, c.nombre, u.nombre
                ORDER BY total_ventas DESC
            ", $params);
            
        case 'ventas_por_vendedor':
            return $db->select("
                SELECT 
                    u.nombre as vendedor,
                    z.nombre as zona,
                    COUNT(v.id) as total_entregas,
                    SUM(v.cantidad_bidones) as total_bidones,
                    SUM(v.total) as total_ventas,
                    COUNT(DISTINCT c.id) as clientes_atendidos,
                    AVG(v.total) as promedio_por_venta
                FROM ventas v
                INNER JOIN clientes c ON v.cliente_id = c.id
                INNER JOIN usuarios u ON c.vendedor_id = u.id
                LEFT JOIN zonas z ON u.zona_id = z.id
                $whereClause
                GROUP BY u.id, u.nombre, z.nombre
                ORDER BY total_ventas DESC
            ", $params);
            
        case 'ventas_diarias':
            return $db->select("
                SELECT 
                    v.fecha,
                    COUNT(v.id) as total_entregas,
                    SUM(v.cantidad_bidones) as total_bidones,
                    SUM(v.total) as total_ventas,
                    COUNT(DISTINCT c.id) as clientes_diferentes
                FROM ventas v
                INNER JOIN clientes c ON v.cliente_id = c.id
                INNER JOIN usuarios u ON c.vendedor_id = u.id
                $whereClause
                GROUP BY v.fecha
                ORDER BY v.fecha DESC
            ", $params);
    }
    
    return [];
}

// Función para exportar a Excel
function exportarExcel($datos, $tipoReporte, $fechaInicio, $fechaFin) {
    header('Content-Type: application/vnd.ms-excel');
    header('Content-Disposition: attachment; filename="reporte_' . $tipoReporte . '_' . date('Y-m-d') . '.xls"');
    
    echo "<html><head><meta charset='UTF-8'></head><body>";
    echo "<h2>Reporte: " . ucwords(str_replace('_', ' ', $tipoReporte)) . "</h2>";
    echo "<p>Período: " . date('d/m/Y', strtotime($fechaInicio)) . " - " . date('d/m/Y', strtotime($fechaFin)) . "</p>";
    echo "<p>Generado: " . date('d/m/Y H:i:s') . "</p><br>";
    
    if (!empty($datos)) {
        echo "<table border='1'>";
        
        // Encabezados
        echo "<tr>";
        foreach (array_keys($datos[0]) as $header) {
            echo "<th>" . ucwords(str_replace('_', ' ', $header)) . "</th>";
        }
        echo "</tr>";
        
        // Datos
        foreach ($datos as $fila) {
            echo "<tr>";
            foreach ($fila as $valor) {
                echo "<td>" . htmlspecialchars($valor ?? '') . "</td>";
            }
            echo "</tr>";
        }
        
        echo "</table>";
    } else {
        echo "<p>No hay datos para mostrar en el período seleccionado.</p>";
    }
    
    echo "</body></html>";
}

// Función para exportar a PDF
function exportarPDF($datos, $tipoReporte, $fechaInicio, $fechaFin) {
    require_once '../../includes/tcpdf/tcpdf.php';
    
    $pdf = new TCPDF(PDF_PAGE_ORIENTATION, PDF_UNIT, PDF_PAGE_FORMAT, true, 'UTF-8', false);
    
    $pdf->SetCreator(PDF_CREATOR);
    $pdf->SetAuthor('Sistema de Agua Mineral');
    $pdf->SetTitle('Reporte ' . ucwords(str_replace('_', ' ', $tipoReporte)));
    
    $pdf->SetHeaderData('', 0, 'Sistema de Agua Mineral', 'Reporte: ' . ucwords(str_replace('_', ' ', $tipoReporte)));
    
    $pdf->setHeaderFont(Array(PDF_FONT_NAME_MAIN, '', PDF_FONT_SIZE_MAIN));
    $pdf->setFooterFont(Array(PDF_FONT_NAME_DATA, '', PDF_FONT_SIZE_DATA));
    
    $pdf->SetDefaultMonospacedFont(PDF_FONT_MONOSPACED);
    
    $pdf->SetMargins(PDF_MARGIN_LEFT, PDF_MARGIN_TOP, PDF_MARGIN_RIGHT);
    $pdf->SetHeaderMargin(PDF_MARGIN_HEADER);
    $pdf->SetFooterMargin(PDF_MARGIN_FOOTER);
    
    $pdf->SetAutoPageBreak(TRUE, PDF_MARGIN_BOTTOM);
    
    $pdf->AddPage();
    
    $html = '<h2>Reporte: ' . ucwords(str_replace('_', ' ', $tipoReporte)) . '</h2>';
    $html .= '<p>Período: ' . date('d/m/Y', strtotime($fechaInicio)) . ' - ' . date('d/m/Y', strtotime($fechaFin)) . '</p>';
    $html .= '<p>Generado: ' . date('d/m/Y H:i:s') . '</p><br>';
    
    if (!empty($datos)) {
        $html .= '<table border="1" cellpadding="4">';
        
        // Encabezados
        $html .= '<tr style="background-color: #f0f0f0;">';
        foreach (array_keys($datos[0]) as $header) {
            $html .= '<th>' . ucwords(str_replace('_', ' ', $header)) . '</th>';
        }
        $html .= '</tr>';
        
        // Datos
        foreach ($datos as $fila) {
            $html .= '<tr>';
            foreach ($fila as $valor) {
                $html .= '<td>' . htmlspecialchars($valor ?? '') . '</td>';
            }
            $html .= '</tr>';
        }
        
        $html .= '</table>';
    } else {
        $html .= '<p>No hay datos para mostrar en el período seleccionado.</p>';
    }
    
    $pdf->writeHTML($html, true, false, true, false, '');
    
    $pdf->Output('reporte_' . $tipoReporte . '_' . date('Y-m-d') . '.pdf', 'D');
}

include '../../includes/header.php';
?>

<div class="row">
    <div class="col-12">
        <h1 class="h3 mb-4">
            <i class="fas fa-chart-bar me-2"></i>Reportes y Exportación
        </h1>
    </div>
</div>

<!-- Filtros -->
<div class="card mb-4">
    <div class="card-header">
        <h5 class="mb-0">Filtros de Reporte</h5>
    </div>
    <div class="card-body">
        <form method="GET" class="row g-3">
            <div class="col-md-3">
                <label for="tipo_reporte" class="form-label">Tipo de Reporte</label>
                <select name="tipo_reporte" id="tipo_reporte" class="form-select">
                    <option value="ventas_detalle" <?= $tipoReporte === 'ventas_detalle' ? 'selected' : '' ?>>
                        Ventas Detalladas
                    </option>
                    <option value="ventas_por_cliente" <?= $tipoReporte === 'ventas_por_cliente' ? 'selected' : '' ?>>
                        Ventas por Cliente
                    </option>
                    <option value="ventas_por_vendedor" <?= $tipoReporte === 'ventas_por_vendedor' ? 'selected' : '' ?>>
                        Ventas por Vendedor
                    </option>
                    <option value="ventas_diarias" <?= $tipoReporte === 'ventas_diarias' ? 'selected' : '' ?>>
                        Ventas Diarias
                    </option>
                </select>
            </div>
            
            <div class="col-md-2">
                <label for="fecha_inicio" class="form-label">Fecha Inicio</label>
                <input type="date" name="fecha_inicio" id="fecha_inicio" class="form-control" 
                       value="<?= $fechaInicio ?>">
            </div>
            
            <div class="col-md-2">
                <label for="fecha_fin" class="form-label">Fecha Fin</label>
                <input type="date" name="fecha_fin" id="fecha_fin" class="form-control" 
                       value="<?= $fechaFin ?>">
            </div>
            
            <div class="col-md-2">
                <label for="vendedor_id" class="form-label">Vendedor</label>
                <select name="vendedor_id" id="vendedor_id" class="form-select">
                    <option value="">Todos los vendedores</option>
                    <?php foreach ($vendedores as $vendedor): ?>
                        <option value="<?= $vendedor['id'] ?>" <?= $vendedorId == $vendedor['id'] ? 'selected' : '' ?>>
                            <?= htmlspecialchars($vendedor['nombre']) ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <div class="col-md-3 d-flex align-items-end">
                <button type="submit" class="btn btn-primary me-2">
                    <i class="fas fa-search me-1"></i>Generar Reporte
                </button>
                <a href="reportes.php" class="btn btn-outline-secondary">
                    <i class="fas fa-times me-1"></i>Limpiar
                </a>
            </div>
        </form>
    </div>
</div>

<!-- Estadísticas del reporte -->
<?php if (!empty($datosReporte)): ?>
<div class="row mb-4">
    <?php
    $totalRegistros = count($datosReporte);
    $totalVentas = 0;
    $totalBidones = 0;
    
    foreach ($datosReporte as $fila) {
        if (isset($fila['total_ventas'])) {
            $totalVentas += $fila['total_ventas'];
        } elseif (isset($fila['total'])) {
            $totalVentas += $fila['total'];
        }
        
        if (isset($fila['total_bidones'])) {
            $totalBidones += $fila['total_bidones'];
        } elseif (isset($fila['cantidad_bidones'])) {
            $totalBidones += $fila['cantidad_bidones'];
        }
    }
    ?>
    
    <div class="col-md-3">
        <div class="card bg-primary text-white">
            <div class="card-body text-center">
                <i class="fas fa-list fa-2x mb-2"></i>
                <h4><?= number_format($totalRegistros) ?></h4>
                <small>Total Registros</small>
            </div>
        </div>
    </div>
    
    <div class="col-md-3">
        <div class="card bg-success text-white">
            <div class="card-body text-center">
                <i class="fas fa-tint fa-2x mb-2"></i>
                <h4><?= number_format($totalBidones) ?></h4>
                <small>Total Bidones</small>
            </div>
        </div>
    </div>
    
    <div class="col-md-3">
        <div class="card bg-info text-white">
            <div class="card-body text-center">
                <i class="fas fa-dollar-sign fa-2x mb-2"></i>
                <h4><?= formatCurrency($totalVentas) ?></h4>
                <small>Total Ventas</small>
            </div>
        </div>
    </div>
    
    <div class="col-md-3">
        <div class="card bg-warning text-white">
            <div class="card-body text-center">
                <i class="fas fa-calculator fa-2x mb-2"></i>
                <h4><?= $totalBidones > 0 ? formatCurrency($totalVentas / $totalBidones) : '$0.00' ?></h4>
                <small>Precio Promedio</small>
            </div>
        </div>
    </div>
</div>
<?php endif; ?>

<!-- Resultados del reporte -->
<div class="card">
    <div class="card-header d-flex justify-content-between align-items-center">
        <h5 class="mb-0">
            Resultados: <?= ucwords(str_replace('_', ' ', $tipoReporte)) ?>
        </h5>
        
        <?php if (!empty($datosReporte)): ?>
        <div class="btn-group">
            <a href="?<?= http_build_query(array_merge($_GET, ['export' => 'excel'])) ?>" 
               class="btn btn-success btn-sm">
                <i class="fas fa-file-excel me-1"></i>Excel
            </a>
            <a href="?<?= http_build_query(array_merge($_GET, ['export' => 'pdf'])) ?>" 
               class="btn btn-danger btn-sm">
                <i class="fas fa-file-pdf me-1"></i>PDF
            </a>
        </div>
        <?php endif; ?>
    </div>
    
    <div class="card-body">
        <?php if (empty($datosReporte)): ?>
            <div class="text-center py-5">
                <i class="fas fa-chart-bar fa-3x text-muted mb-3"></i>
                <h5 class="text-muted">No hay datos para mostrar</h5>
                <p class="text-muted">No se encontraron registros con los filtros aplicados.</p>
            </div>
        <?php else: ?>
            <div class="table-responsive">
                <table class="table table-striped table-hover" id="reporteTable">
                    <thead class="table-dark">
                        <tr>
                            <?php foreach (array_keys($datosReporte[0]) as $header): ?>
                                <th><?= ucwords(str_replace('_', ' ', $header)) ?></th>
                            <?php endforeach; ?>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($datosReporte as $fila): ?>
                            <tr>
                                <?php foreach ($fila as $key => $valor): ?>
                                    <td>
                                        <?php if (strpos($key, 'fecha') !== false && $valor): ?>
                                            <?= formatDate($valor) ?>
                                        <?php elseif (strpos($key, 'total') !== false || strpos($key, 'precio') !== false || strpos($key, 'promedio') !== false): ?>
                                            <?= formatCurrency($valor ?? 0) ?>
                                        <?php else: ?>
                                            <?= htmlspecialchars($valor ?? '') ?>
                                        <?php endif; ?>
                                    </td>
                                <?php endforeach; ?>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>
</div>

<!-- Gráfico de ventas por período -->
<?php if ($tipoReporte === 'ventas_diarias' && !empty($datosReporte)): ?>
<div class="row mt-4">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Gráfico de Ventas Diarias</h5>
            </div>
            <div class="card-body">
                <canvas id="ventasDiariasChart" width="400" height="100"></canvas>
            </div>
        </div>
    </div>
</div>
<?php endif; ?>

<?php
$extraJS = '
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
// DataTable para mejor navegación
$(document).ready(function() {
    $("#reporteTable").DataTable({
        "language": {
            "url": "//cdn.datatables.net/plug-ins/1.10.24/i18n/Spanish.json"
        },
        "pageLength": 25,
        "order": [],
        "dom": "Bfrtip",
        "buttons": [
            "copy", "csv", "print"
        ]
    });
});

// Gráfico de ventas diarias
' . ($tipoReporte === 'ventas_diarias' && !empty($datosReporte) ? '
const ventasDiariasData = ' . json_encode($datosReporte) . ';

const ctx = document.getElementById("ventasDiariasChart").getContext("2d");

const labels = ventasDiariasData.map(item => {
    const date = new Date(item.fecha);
    return date.toLocaleDateString("es-ES", { month: "short", day: "numeric" });
});

const bidones = ventasDiariasData.map(item => parseInt(item.total_bidones));
const ventas = ventasDiariasData.map(item => parseFloat(item.total_ventas));

new Chart(ctx, {
    type: "line",
    data: {
        labels: labels,
        datasets: [{
            label: "Bidones Vendidos",
            data: bidones,
            borderColor: "rgb(54, 162, 235)",
            backgroundColor: "rgba(54, 162, 235, 0.1)",
            yAxisID: "y"
        }, {
            label: "Ventas ($)",
            data: ventas,
            borderColor: "rgb(255, 99, 132)",
            backgroundColor: "rgba(255, 99, 132, 0.1)",
            yAxisID: "y1"
        }]
    },
    options: {
        responsive: true,
        interaction: {
            mode: "index",
            intersect: false,
        },
        scales: {
            y: {
                type: "linear",
                display: true,
                position: "left",
                title: {
                    display: true,
                    text: "Bidones"
                }
            },
            y1: {
                type: "linear",
                display: true,
                position: "right",
                title: {
                    display: true,
                    text: "Ventas ($)"
                },
                grid: {
                    drawOnChartArea: false,
                },
            }
        }
    }
});
' : '') . '

// Atajos de teclado para fechas
document.addEventListener("keydown", function(e) {
    if (e.ctrlKey) {
        switch(e.key) {
            case "1": // Ctrl+1: Este mes
                document.getElementById("fecha_inicio").value = "' . date('Y-m-01') . '";
                document.getElementById("fecha_fin").value = "' . date('Y-m-t') . '";
                break;
            case "2": // Ctrl+2: Mes anterior
                document.getElementById("fecha_inicio").value = "' . date('Y-m-01', strtotime('-1 month')) . '";
                document.getElementById("fecha_fin").value = "' . date('Y-m-t', strtotime('-1 month')) . '";
                break;
            case "3": // Ctrl+3: Últimos 7 días
                document.getElementById("fecha_inicio").value = "' . date('Y-m-d', strtotime('-7 days')) . '";
                document.getElementById("fecha_fin").value = "' . date('Y-m-d') . '";
                break;
        }
    }
});
</script>

<link rel="stylesheet" type="text/css" href="https://cdn.datatables.net/1.10.24/css/dataTables.bootstrap5.min.css">
<link rel="stylesheet" type="text/css" href="https://cdn.datatables.net/buttons/1.7.0/css/buttons.bootstrap5.min.css">
<script type="text/javascript" charset="utf8" src="https://cdn.datatables.net/1.10.24/js/jquery.dataTables.js"></script>
<script type="text/javascript" charset="utf8" src="https://cdn.datatables.net/1.10.24/js/dataTables.bootstrap5.min.js"></script>
<script type="text/javascript" charset="utf8" src="https://cdn.datatables.net/buttons/1.7.0/js/dataTables.buttons.min.js"></script>
<script type="text/javascript" charset="utf8" src="https://cdn.datatables.net/buttons/1.7.0/js/buttons.bootstrap5.min.js"></script>
<script type="text/javascript" charset="utf8" src="https://cdnjs.cloudflare.com/ajax/libs/jszip/3.1.3/jszip.min.js"></script>
<script type="text/javascript" charset="utf8" src="https://cdn.datatables.net/buttons/1.7.0/js/buttons.html5.min.js"></script>
<script type="text/javascript" charset="utf8" src="https://cdn.datatables.net/buttons/1.7.0/js/buttons.print.min.js"></script>
';

include '../../includes/footer.php';
?>
