<?php
/**
 * Gestión de Ubicación GPS de Clientes - Administrador
 * Sistema de Gestión de Agua Mineral
 */

require_once '../../includes/init.php';

// Verificar que sea administrador
checkPermission('administrador');

$pageTitle = 'Ubicación GPS de Clientes - Administrador';

$db = Database::getInstance();

$clienteId = $_GET['id'] ?? 0;

// Obtener información del cliente
$cliente = $db->select("
    SELECT c.*, u.nombre as vendedor_nombre 
    FROM clientes c 
    INNER JOIN usuarios u ON c.vendedor_id = u.id 
    WHERE c.id = ?
", [$clienteId]);

if (empty($cliente)) {
    setFlashMessage('Cliente no encontrado.', 'error');
    redirect('clientes.php');
}

$cliente = $cliente[0];

// Procesar formulario
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $latitud = $_POST['latitud'] ?? '';
    $longitud = $_POST['longitud'] ?? '';
    $direccion = sanitize($_POST['direccion']);
    
    // Validar coordenadas
    if (!empty($latitud) && !empty($longitud)) {
        $latitud = (float)$latitud;
        $longitud = (float)$longitud;
        
        // Validar rangos aproximados para Paraguay
        if ($latitud >= -27.6 && $latitud <= -19.3 && $longitud >= -62.6 && $longitud <= -54.3) {
            $query = "UPDATE clientes SET latitud = ?, longitud = ?, direccion = ? WHERE id = ?";
            
            if ($db->execute($query, [$latitud, $longitud, $direccion, $clienteId])) {
                setFlashMessage('Ubicación GPS actualizada correctamente.', 'success');
                logActivity($_SESSION['user_id'], 'update_client_location', "Ubicación GPS actualizada para cliente ID $clienteId");
                redirect('clientes.php');
            } else {
                setFlashMessage('Error al actualizar la ubicación.', 'error');
            }
        } else {
            setFlashMessage('Las coordenadas no son válidas para Paraguay.', 'error');
        }
    } else {
        // Limpiar coordenadas si están vacías
        $query = "UPDATE clientes SET latitud = NULL, longitud = NULL, direccion = ? WHERE id = ?";
        
        if ($db->execute($query, [$direccion, $clienteId])) {
            setFlashMessage('Ubicación GPS eliminada correctamente.', 'success');
            redirect('clientes.php');
        } else {
            setFlashMessage('Error al actualizar la información.', 'error');
        }
    }
}

include '../../includes/header.php';
?>

<div class="row">
    <div class="col-12">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h1 class="h3">
                <i class="fas fa-map-marker-alt me-2"></i>Ubicación GPS del Cliente
            </h1>
            <a href="clientes.php" class="btn btn-secondary">
                <i class="fas fa-arrow-left me-2"></i>Volver a Clientes
            </a>
        </div>
    </div>
</div>

<!-- Información del cliente -->
<div class="card mb-4">
    <div class="card-body">
        <div class="row">
            <div class="col-md-6">
                <h5><?= htmlspecialchars($cliente['nombre']) ?></h5>
                <p class="text-muted mb-1">
                    <i class="fas fa-user-tie me-1"></i>
                    Vendedor: <?= htmlspecialchars($cliente['vendedor_nombre']) ?>
                </p>
                <?php if ($cliente['telefono']): ?>
                    <p class="text-muted mb-1">
                        <i class="fas fa-phone me-1"></i>
                        <?= htmlspecialchars($cliente['telefono']) ?>
                    </p>
                <?php endif; ?>
            </div>
            <div class="col-md-6">
                <?php if ($cliente['latitud'] && $cliente['longitud']): ?>
                    <div class="alert alert-success">
                        <i class="fas fa-check-circle me-1"></i>
                        <strong>Ubicación GPS configurada</strong><br>
                        Lat: <?= $cliente['latitud'] ?>, Lng: <?= $cliente['longitud'] ?>
                    </div>
                <?php else: ?>
                    <div class="alert alert-warning">
                        <i class="fas fa-exclamation-triangle me-1"></i>
                        <strong>Sin ubicación GPS</strong><br>
                        Configure las coordenadas para mostrar en el mapa
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<div class="row">
    <!-- Formulario -->
    <div class="col-lg-4">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Configurar Ubicación</h5>
            </div>
            <div class="card-body">
                <form method="POST" id="ubicacionForm">
                    <div class="mb-3">
                        <label for="direccion" class="form-label">Dirección</label>
                        <textarea name="direccion" id="direccion" class="form-control" rows="3" required><?= htmlspecialchars($cliente['direccion']) ?></textarea>
                        <div class="form-text">Dirección completa del cliente</div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="latitud" class="form-label">Latitud</label>
                        <input type="number" name="latitud" id="latitud" class="form-control" 
                               step="0.000001" min="-27.6" max="-19.3" 
                               value="<?= $cliente['latitud'] ?>" placeholder="-25.263740">
                        <div class="form-text">Coordenada de latitud (ej: -25.263740)</div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="longitud" class="form-label">Longitud</label>
                        <input type="number" name="longitud" id="longitud" class="form-control" 
                               step="0.000001" min="-62.6" max="-54.3" 
                               value="<?= $cliente['longitud'] ?>" placeholder="-57.575926">
                        <div class="form-text">Coordenada de longitud (ej: -57.575926)</div>
                    </div>
                    
                    <div class="mb-3">
                        <button type="button" class="btn btn-info w-100" onclick="buscarDireccion()">
                            <i class="fas fa-search me-1"></i>Buscar por Dirección
                        </button>
                    </div>
                    
                    <div class="mb-3">
                        <button type="button" class="btn btn-success w-100" onclick="obtenerUbicacionActual()">
                            <i class="fas fa-crosshairs me-1"></i>Mi Ubicación Actual
                        </button>
                    </div>
                    
                    <hr>
                    
                    <div class="d-grid gap-2">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save me-1"></i>Guardar Ubicación
                        </button>
                        
                        <?php if ($cliente['latitud'] && $cliente['longitud']): ?>
                            <button type="button" class="btn btn-outline-danger" onclick="limpiarUbicacion()">
                                <i class="fas fa-trash me-1"></i>Eliminar Ubicación
                            </button>
                        <?php endif; ?>
                    </div>
                </form>
            </div>
        </div>
        
        <!-- Instrucciones -->
        <div class="card mt-3">
            <div class="card-header">
                <h6 class="mb-0">Instrucciones</h6>
            </div>
            <div class="card-body">
                <ol class="small">
                    <li>Ingrese la dirección completa del cliente</li>
                    <li>Use "Buscar por Dirección" para obtener coordenadas automáticamente</li>
                    <li>O ingrese las coordenadas manualmente</li>
                    <li>Verifique la ubicación en el mapa</li>
                    <li>Haga clic en el mapa para ajustar la posición</li>
                    <li>Guarde los cambios</li>
                </ol>
            </div>
        </div>
    </div>
    
    <!-- Mapa -->
    <div class="col-lg-8">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Mapa Interactivo</h5>
            </div>
            <div class="card-body p-0">
                <div id="map" style="height: 500px; width: 100%;"></div>
            </div>
            <div class="card-footer">
                <small class="text-muted">
                    <i class="fas fa-info-circle me-1"></i>
                    Haga clic en el mapa para establecer la ubicación exacta del cliente
                </small>
            </div>
        </div>
    </div>
</div>

<?php
$extraJS = '
<script>
let map;
let marker;
let geocoder;

function initMap() {
    // Centro por defecto (Asunción, Paraguay)
    const defaultCenter = { lat: -25.2637, lng: -57.5759 };
    
    // Usar ubicación del cliente si existe
    const clienteLat = ' . ($cliente['latitud'] ?? 'null') . ';
    const clienteLng = ' . ($cliente['longitud'] ?? 'null') . ';
    
    const center = (clienteLat && clienteLng) ? 
        { lat: clienteLat, lng: clienteLng } : defaultCenter;
    
    map = new google.maps.Map(document.getElementById("map"), {
        zoom: 15,
        center: center,
        mapTypeId: google.maps.MapTypeId.ROADMAP
    });
    
    geocoder = new google.maps.Geocoder();
    
    // Crear marcador
    marker = new google.maps.Marker({
        position: center,
        map: map,
        draggable: true,
        title: "' . htmlspecialchars($cliente['nombre']) . '"
    });
    
    // Si no hay ubicación del cliente, ocultar marcador inicialmente
    if (!clienteLat || !clienteLng) {
        marker.setVisible(false);
    }
    
    // Eventos del mapa
    map.addListener("click", (event) => {
        colocarMarcador(event.latLng);
    });
    
    marker.addListener("dragend", (event) => {
        actualizarCoordenadas(event.latLng.lat(), event.latLng.lng());
    });
}

function colocarMarcador(location) {
    marker.setPosition(location);
    marker.setVisible(true);
    actualizarCoordenadas(location.lat(), location.lng());
}

function actualizarCoordenadas(lat, lng) {
    document.getElementById("latitud").value = lat.toFixed(6);
    document.getElementById("longitud").value = lng.toFixed(6);
}

function buscarDireccion() {
    const direccion = document.getElementById("direccion").value;
    
    if (!direccion.trim()) {
        alert("Por favor, ingrese una dirección para buscar.");
        return;
    }
    
    geocoder.geocode({ address: direccion + ", Paraguay" }, (results, status) => {
        if (status === "OK" && results[0]) {
            const location = results[0].geometry.location;
            
            map.setCenter(location);
            map.setZoom(16);
            
            colocarMarcador(location);
            
            // Actualizar dirección con el resultado de Google
            document.getElementById("direccion").value = results[0].formatted_address;
        } else {
            alert("No se pudo encontrar la dirección. Intente con una dirección más específica.");
        }
    });
}

function obtenerUbicacionActual() {
    if (navigator.geolocation) {
        navigator.geolocation.getCurrentPosition(
            (position) => {
                const location = {
                    lat: position.coords.latitude,
                    lng: position.coords.longitude
                };
                
                map.setCenter(location);
                map.setZoom(16);
                
                colocarMarcador(new google.maps.LatLng(location.lat, location.lng));
            },
            (error) => {
                alert("No se pudo obtener su ubicación actual: " + error.message);
            }
        );
    } else {
        alert("Su navegador no soporta geolocalización.");
    }
}

function limpiarUbicacion() {
    if (confirm("¿Está seguro que desea eliminar la ubicación GPS de este cliente?")) {
        document.getElementById("latitud").value = "";
        document.getElementById("longitud").value = "";
        marker.setVisible(false);
    }
}

// Validación del formulario
document.getElementById("ubicacionForm").addEventListener("submit", function(e) {
    const lat = document.getElementById("latitud").value;
    const lng = document.getElementById("longitud").value;
    
    if (lat && lng) {
        const latNum = parseFloat(lat);
        const lngNum = parseFloat(lng);
        
        // Validar rangos para Paraguay
        if (latNum < -27.6 || latNum > -19.3 || lngNum < -62.6 || lngNum > -54.3) {
            e.preventDefault();
            alert("Las coordenadas no son válidas para Paraguay. Verifique los valores.");
            return;
        }
    }
});

// Actualizar coordenadas cuando se cambian manualmente
document.getElementById("latitud").addEventListener("change", function() {
    const lat = parseFloat(this.value);
    const lng = parseFloat(document.getElementById("longitud").value);
    
    if (!isNaN(lat) && !isNaN(lng)) {
        const location = new google.maps.LatLng(lat, lng);
        map.setCenter(location);
        colocarMarcador(location);
    }
});

document.getElementById("longitud").addEventListener("change", function() {
    const lat = parseFloat(document.getElementById("latitud").value);
    const lng = parseFloat(this.value);
    
    if (!isNaN(lat) && !isNaN(lng)) {
        const location = new google.maps.LatLng(lat, lng);
        map.setCenter(location);
        colocarMarcador(location);
    }
});
</script>

<script async defer src="https://maps.googleapis.com/maps/api/js?key=' . GOOGLE_MAPS_API_KEY . '&callback=initMap"></script>
';

include '../../includes/footer.php';
?>
