<?php
require_once '../includes/config.php';

// Verificar que el huésped esté registrado
if (!isset($_SESSION['huesped_id'])) {
    redirect('registro.php');
}

$error = '';
$success = '';

$db = getDBConnection();

// Obtener tarifas disponibles
$stmt = $db->query("SELECT * FROM tarifas WHERE activa = 1 ORDER BY precio_por_noche ASC");
$tarifas = $stmt->fetchAll();

// Obtener reservas existentes para marcar fechas ocupadas
$stmt = $db->query("SELECT fecha_entrada, fecha_salida FROM reservas WHERE estado IN ('confirmada', 'en_curso', 'pendiente')");
$reservas_existentes = $stmt->fetchAll();

// Procesar formulario de reserva
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['crear_reserva'])) {
    $fecha_entrada = $_POST['fecha_entrada'];
    $fecha_salida = $_POST['fecha_salida'];
    $cantidad_personas = (int)$_POST['cantidad_personas'];
    $tarifa_id = (int)$_POST['tarifa_id'];
    $comentarios = sanitize($_POST['comentarios'] ?? '');
    
    // Validaciones
    if (empty($fecha_entrada) || empty($fecha_salida) || $cantidad_personas < 1 || $tarifa_id < 1) {
        $error = 'Por favor complete todos los campos obligatorios';
    } else {
        // Calcular días
        $entrada = new DateTime($fecha_entrada);
        $salida = new DateTime($fecha_salida);
        $hoy = new DateTime();
        $hoy->setTime(0, 0, 0);
        
        if ($entrada < $hoy) {
            $error = 'La fecha de entrada no puede ser anterior a hoy';
        } elseif ($salida <= $entrada) {
            $error = 'La fecha de salida debe ser posterior a la fecha de entrada';
        } else {
            $cantidad_dias = $entrada->diff($salida)->days;
            
            // Verificar disponibilidad
            $stmt = $db->prepare("
                SELECT COUNT(*) as conflictos 
                FROM reservas 
                WHERE estado IN ('confirmada', 'en_curso', 'pendiente')
                AND (
                    (fecha_entrada <= ? AND fecha_salida > ?) OR
                    (fecha_entrada < ? AND fecha_salida >= ?) OR
                    (fecha_entrada >= ? AND fecha_salida <= ?)
                )
            ");
            $stmt->execute([$fecha_entrada, $fecha_entrada, $fecha_salida, $fecha_salida, $fecha_entrada, $fecha_salida]);
            $conflictos = $stmt->fetch()['conflictos'];
            
            if ($conflictos > 0) {
                $error = 'Las fechas seleccionadas no están disponibles. Por favor elija otras fechas.';
            } else {
                // Obtener tarifa y calcular costo
                $stmt = $db->prepare("SELECT * FROM tarifas WHERE id = ?");
                $stmt->execute([$tarifa_id]);
                $tarifa = $stmt->fetch();
                
                $costo_hospedaje = ($tarifa['precio_por_noche'] * $cantidad_dias) + 
                                  ($tarifa['precio_por_persona'] * ($cantidad_personas - 1) * $cantidad_dias);
                
                // Crear reserva
                $stmt = $db->prepare("
                    INSERT INTO reservas (usuario_id, fecha_entrada, fecha_salida, cantidad_personas, 
                                         cantidad_dias, tarifa_id, costo_hospedaje, comentarios, estado)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'pendiente')
                ");
                $stmt->execute([
                    $_SESSION['huesped_id'],
                    $fecha_entrada,
                    $fecha_salida,
                    $cantidad_personas,
                    $cantidad_dias,
                    $tarifa_id,
                    $costo_hospedaje,
                    $comentarios
                ]);
                
                $reserva_id = $db->lastInsertId();
                $_SESSION['reserva_id'] = $reserva_id;
                
                // Redirigir a comprobante
                redirect('comprobante.php?id=' . $reserva_id);
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Nueva Reserva - <?php echo SITE_NAME; ?></title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <style>
        .calendar-container {
            margin: 20px 0;
        }
        .date-input-group {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 15px;
        }
        @media (max-width: 768px) {
            .date-input-group {
                grid-template-columns: 1fr;
            }
        }
        .cost-summary {
            background: linear-gradient(135deg, var(--color-primary), var(--color-primary-light));
            color: white;
            padding: 20px;
            border-radius: var(--radius);
            margin-top: 20px;
        }
        .cost-item {
            display: flex;
            justify-content: space-between;
            padding: 10px 0;
            border-bottom: 1px solid rgba(255,255,255,0.2);
        }
        .cost-total {
            font-size: 1.5rem;
            font-weight: bold;
            margin-top: 10px;
        }
    </style>
</head>
<body>
    <!-- Header -->
    <header class="header">
        <div class="header-content">
            <div class="logo-container">
                <img src="../assets/images/logo.png" alt="Logo Posada" class="logo">
                <h1 class="site-title">Nueva Reserva</h1>
            </div>
            <nav>
                <ul class="nav-menu">
                    <li><span style="color: white;">Hola, <?php echo htmlspecialchars($_SESSION['huesped_nombre']); ?></span></li>
                </ul>
            </nav>
        </div>
    </header>

    <div class="container mt-3">
        <div class="grid grid-2" style="align-items: start;">
            <!-- Formulario de reserva -->
            <div class="card">
                <div class="card-header">
                    <h2 class="card-title">Datos de la Reserva</h2>
                </div>
                <div class="card-body">
                    <?php if ($error): ?>
                        <div class="alert alert-danger">
                            <?php echo htmlspecialchars($error); ?>
                        </div>
                    <?php endif; ?>
                    
                    <form method="POST" action="" id="formReserva">
                        <div class="form-group">
                            <label class="form-label">Seleccione Tarifa *</label>
                            <select name="tarifa_id" id="tarifa_id" class="form-control" required>
                                <option value="">-- Seleccione una tarifa --</option>
                                <?php foreach ($tarifas as $tarifa): ?>
                                    <option value="<?php echo $tarifa['id']; ?>" 
                                            data-precio-noche="<?php echo $tarifa['precio_por_noche']; ?>"
                                            data-precio-persona="<?php echo $tarifa['precio_por_persona']; ?>">
                                        <?php echo htmlspecialchars($tarifa['nombre']); ?> - 
                                        <?php echo formatMoney($tarifa['precio_por_noche']); ?>/noche
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="date-input-group">
                            <div class="form-group">
                                <label class="form-label">Fecha de Entrada *</label>
                                <input type="date" name="fecha_entrada" id="fecha_entrada" 
                                       class="form-control" min="<?php echo date('Y-m-d'); ?>" required>
                            </div>
                            
                            <div class="form-group">
                                <label class="form-label">Fecha de Salida *</label>
                                <input type="date" name="fecha_salida" id="fecha_salida" 
                                       class="form-control" min="<?php echo date('Y-m-d', strtotime('+1 day')); ?>" required>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">Cantidad de Personas *</label>
                            <input type="number" name="cantidad_personas" id="cantidad_personas" 
                                   class="form-control" min="1" max="10" value="1" required>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">Comentarios o Solicitudes Especiales</label>
                            <textarea name="comentarios" id="comentarios" class="form-control" 
                                      rows="4" placeholder="Ej: Necesito cuna para bebé, dieta especial, etc."></textarea>
                        </div>
                        
                        <div class="alert alert-info">
                            <strong>Nota:</strong> Su reserva quedará en estado "Pendiente" hasta que el administrador la confirme.
                        </div>
                        
                        <button type="submit" name="crear_reserva" class="btn btn-primary btn-block btn-lg">
                            Crear Reserva
                        </button>
                    </form>
                </div>
            </div>
            
            <!-- Resumen de costos -->
            <div>
                <div class="card">
                    <div class="card-header">
                        <h3 class="card-title">Resumen de Costos</h3>
                    </div>
                    <div class="card-body">
                        <div id="costoResumen">
                            <p style="color: var(--color-text-light); text-align: center; padding: 20px;">
                                Complete los datos de la reserva para ver el costo estimado
                            </p>
                        </div>
                    </div>
                </div>
                
                <div class="card mt-2">
                    <div class="card-header">
                        <h3 class="card-title">Fechas Ocupadas</h3>
                    </div>
                    <div class="card-body">
                        <p style="color: var(--color-text-light); font-size: 0.9rem;">
                            Las siguientes fechas ya están reservadas:
                        </p>
                        <div id="fechasOcupadas" style="max-height: 200px; overflow-y: auto;">
                            <?php if (!empty($reservas_existentes)): ?>
                                <?php foreach ($reservas_existentes as $reserva): ?>
                                    <div style="padding: 8px; background: var(--color-bg); border-radius: 4px; margin-bottom: 5px; font-size: 0.9rem;">
                                        <?php echo formatDate($reserva['fecha_entrada']); ?> - 
                                        <?php echo formatDate($reserva['fecha_salida']); ?>
                                    </div>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <p style="color: var(--color-success);">✓ Todas las fechas están disponibles</p>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <footer class="footer">
        <div class="container">
            <p>&copy; <?php echo date('Y'); ?> Sistema de Reservas para Posadas. Todos los derechos reservados.</p>
        </div>
    </footer>

    <script src="../assets/js/main.js"></script>
    <script>
        // Calcular costo en tiempo real
        const tarifaSelect = document.getElementById('tarifa_id');
        const fechaEntrada = document.getElementById('fecha_entrada');
        const fechaSalida = document.getElementById('fecha_salida');
        const cantidadPersonas = document.getElementById('cantidad_personas');
        const costoResumen = document.getElementById('costoResumen');
        
        function calcularCosto() {
            const tarifaOption = tarifaSelect.options[tarifaSelect.selectedIndex];
            if (!tarifaOption || !tarifaOption.value) return;
            
            const precioNoche = parseFloat(tarifaOption.dataset.precioNoche);
            const precioPorPersona = parseFloat(tarifaOption.dataset.precioPersona);
            const personas = parseInt(cantidadPersonas.value) || 1;
            
            if (!fechaEntrada.value || !fechaSalida.value) return;
            
            const entrada = new Date(fechaEntrada.value);
            const salida = new Date(fechaSalida.value);
            const dias = Math.ceil((salida - entrada) / (1000 * 60 * 60 * 24));
            
            if (dias <= 0) {
                costoResumen.innerHTML = '<p style="color: var(--color-danger);">La fecha de salida debe ser posterior a la entrada</p>';
                return;
            }
            
            const costoNoches = precioNoche * dias;
            const costoPersonasAdicionales = precioPorPersona * (personas - 1) * dias;
            const costoTotal = costoNoches + costoPersonasAdicionales;
            
            costoResumen.innerHTML = `
                <div class="cost-summary">
                    <div class="cost-item">
                        <span>Tarifa seleccionada:</span>
                        <span>${tarifaOption.text.split(' - ')[0]}</span>
                    </div>
                    <div class="cost-item">
                        <span>Cantidad de noches:</span>
                        <span>${dias} ${dias === 1 ? 'noche' : 'noches'}</span>
                    </div>
                    <div class="cost-item">
                        <span>Costo por noches:</span>
                        <span>${formatCurrency(costoNoches)}</span>
                    </div>
                    ${personas > 1 ? `
                    <div class="cost-item">
                        <span>Personas adicionales (${personas - 1}):</span>
                        <span>${formatCurrency(costoPersonasAdicionales)}</span>
                    </div>
                    ` : ''}
                    <div class="cost-item cost-total">
                        <span>TOTAL:</span>
                        <span>${formatCurrency(costoTotal)}</span>
                    </div>
                </div>
            `;
        }
        
        tarifaSelect.addEventListener('change', calcularCosto);
        fechaEntrada.addEventListener('change', calcularCosto);
        fechaSalida.addEventListener('change', calcularCosto);
        cantidadPersonas.addEventListener('input', calcularCosto);
        
        // Actualizar fecha mínima de salida cuando cambia la entrada
        fechaEntrada.addEventListener('change', function() {
            const entrada = new Date(this.value);
            entrada.setDate(entrada.getDate() + 1);
            fechaSalida.min = entrada.toISOString().split('T')[0];
        });
        
        // Validación del formulario
        document.getElementById('formReserva').addEventListener('submit', function(e) {
            if (!validateDates(fechaEntrada.value, fechaSalida.value)) {
                e.preventDefault();
                return false;
            }
        });
    </script>
</body>
</html>
