<?php
require_once '../includes/config.php';

if (!isLoggedIn() || !isAdmin()) {
    redirect('login.php');
}

$reserva_id = (int)($_GET['reserva_id'] ?? 0);
if (!$reserva_id) {
    redirect('reservas.php');
}

$db = getDBConnection();

// Obtener datos de la reserva
$stmt = $db->prepare("
    SELECT r.*, u.nombre_completo, u.cedula_pasaporte
    FROM reservas r
    JOIN usuarios u ON r.usuario_id = u.id
    WHERE r.id = ?
");
$stmt->execute([$reserva_id]);
$reserva = $stmt->fetch();

if (!$reserva) {
    redirect('reservas.php');
}

// Agregar consumo
if (isset($_POST['agregar_consumo'])) {
    $producto_id = (int)$_POST['producto_id'];
    $cantidad = (int)$_POST['cantidad'];
    
    // Obtener precio del producto
    $stmt = $db->prepare("SELECT precio FROM productos WHERE id = ?");
    $stmt->execute([$producto_id]);
    $producto = $stmt->fetch();
    
    if ($producto) {
        $precio_unitario = $producto['precio'];
        $subtotal = $precio_unitario * $cantidad;
        
        $stmt = $db->prepare("INSERT INTO consumos (reserva_id, producto_id, cantidad, precio_unitario, subtotal) VALUES (?, ?, ?, ?, ?)");
        $stmt->execute([$reserva_id, $producto_id, $cantidad, $precio_unitario, $subtotal]);
    }
}

// Eliminar consumo
if (isset($_GET['eliminar_consumo'])) {
    $consumo_id = (int)$_GET['eliminar_consumo'];
    $stmt = $db->prepare("DELETE FROM consumos WHERE id = ?");
    $stmt->execute([$consumo_id]);
}

// Obtener consumos de la reserva
$stmt = $db->prepare("
    SELECT c.*, p.nombre as producto_nombre, cat.nombre as categoria_nombre
    FROM consumos c
    JOIN productos p ON c.producto_id = p.id
    JOIN categorias_productos cat ON p.categoria_id = cat.id
    WHERE c.reserva_id = ?
    ORDER BY c.fecha_consumo DESC
");
$stmt->execute([$reserva_id]);
$consumos = $stmt->fetchAll();

// Calcular total de consumos
$total_consumos = array_sum(array_column($consumos, 'subtotal'));
$total_general = $reserva['costo_hospedaje'] + $total_consumos;

// Obtener productos disponibles
$stmt = $db->query("
    SELECT p.*, c.nombre as categoria_nombre
    FROM productos p
    JOIN categorias_productos c ON p.categoria_id = c.id
    WHERE p.disponible = 1
    ORDER BY c.nombre, p.nombre
");
$productos = $stmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gestión de Gastos - <?php echo SITE_NAME; ?></title>
    <link rel="stylesheet" href="../assets/css/style.css">
</head>
<body>
    <header class="header">
        <div class="header-content">
            <div class="logo-container">
                <img src="../assets/images/logo.png" alt="Logo Posada" class="logo">
                <h1 class="site-title">Gestión de Gastos</h1>
            </div>
            <nav>
                <ul class="nav-menu">
                    <li><a href="reservas.php">← Volver a Reservas</a></li>
                    <li><a href="logout.php">Cerrar Sesión</a></li>
                </ul>
            </nav>
        </div>
    </header>

    <div class="container mt-3">
        <div class="grid grid-2" style="align-items: start;">
            <!-- Información de la reserva -->
            <div>
                <div class="card">
                    <div class="card-header">
                        <h3 class="card-title">Información de la Reserva</h3>
                    </div>
                    <div class="card-body">
                        <p><strong>Reserva:</strong> #<?php echo str_pad($reserva['id'], 6, '0', STR_PAD_LEFT); ?></p>
                        <p><strong>Huésped:</strong> <?php echo htmlspecialchars($reserva['nombre_completo']); ?></p>
                        <p><strong>CI/Pasaporte:</strong> <?php echo htmlspecialchars($reserva['cedula_pasaporte']); ?></p>
                        <p><strong>Entrada:</strong> <?php echo formatDate($reserva['fecha_entrada']); ?></p>
                        <p><strong>Salida:</strong> <?php echo formatDate($reserva['fecha_salida']); ?></p>
                        <p><strong>Personas:</strong> <?php echo $reserva['cantidad_personas']; ?></p>
                    </div>
                </div>

                <!-- Resumen de costos -->
                <div class="card mt-2">
                    <div class="card-header">
                        <h3 class="card-title">Resumen de Costos</h3>
                    </div>
                    <div class="card-body">
                        <div style="display: flex; justify-content: space-between; padding: 10px 0; border-bottom: 1px solid var(--color-border);">
                            <span>Costo de Hospedaje:</span>
                            <strong><?php echo formatMoney($reserva['costo_hospedaje']); ?></strong>
                        </div>
                        <div style="display: flex; justify-content: space-between; padding: 10px 0; border-bottom: 1px solid var(--color-border);">
                            <span>Total Consumos:</span>
                            <strong><?php echo formatMoney($total_consumos); ?></strong>
                        </div>
                        <div style="display: flex; justify-content: space-between; padding: 15px 0; font-size: 1.3rem;">
                            <span><strong>TOTAL GENERAL:</strong></span>
                            <strong style="color: var(--color-primary);"><?php echo formatMoney($total_general); ?></strong>
                        </div>
                        <a href="imprimir_gastos.php?reserva_id=<?php echo $reserva_id; ?>" class="btn btn-primary btn-block" target="_blank">
                            📄 Imprimir Detalle de Gastos (PDF)
                        </a>
                    </div>
                </div>

                <!-- Agregar consumo -->
                <div class="card mt-2">
                    <div class="card-header">
                        <h3 class="card-title">Agregar Consumo</h3>
                    </div>
                    <div class="card-body">
                        <form method="POST">
                            <div class="form-group">
                                <label class="form-label">Producto</label>
                                <select name="producto_id" class="form-control" required>
                                    <option value="">-- Seleccione --</option>
                                    <?php 
                                    $categoria_actual = '';
                                    foreach ($productos as $producto): 
                                        if ($categoria_actual != $producto['categoria_nombre']) {
                                            if ($categoria_actual != '') echo '</optgroup>';
                                            echo '<optgroup label="' . htmlspecialchars($producto['categoria_nombre']) . '">';
                                            $categoria_actual = $producto['categoria_nombre'];
                                        }
                                    ?>
                                        <option value="<?php echo $producto['id']; ?>">
                                            <?php echo htmlspecialchars($producto['nombre']); ?> - <?php echo formatMoney($producto['precio']); ?>
                                        </option>
                                    <?php endforeach; ?>
                                    <?php if ($categoria_actual != '') echo '</optgroup>'; ?>
                                </select>
                            </div>
                            <div class="form-group">
                                <label class="form-label">Cantidad</label>
                                <input type="number" name="cantidad" class="form-control" min="1" value="1" required>
                            </div>
                            <button type="submit" name="agregar_consumo" class="btn btn-success btn-block">
                                ➕ Agregar Consumo
                            </button>
                        </form>
                    </div>
                </div>
            </div>

            <!-- Lista de consumos -->
            <div class="card">
                <div class="card-header">
                    <h3 class="card-title">Detalle de Consumos</h3>
                </div>
                <div class="card-body">
                    <?php if (empty($consumos)): ?>
                        <p style="text-align: center; color: var(--color-text-light); padding: 20px;">
                            No hay consumos registrados para esta reserva
                        </p>
                    <?php else: ?>
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>Fecha</th>
                                    <th>Producto</th>
                                    <th>Categoría</th>
                                    <th>Cant.</th>
                                    <th>P. Unit.</th>
                                    <th>Subtotal</th>
                                    <th>Acción</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($consumos as $consumo): ?>
                                    <tr>
                                        <td><?php echo date('d/m H:i', strtotime($consumo['fecha_consumo'])); ?></td>
                                        <td><?php echo htmlspecialchars($consumo['producto_nombre']); ?></td>
                                        <td><?php echo htmlspecialchars($consumo['categoria_nombre']); ?></td>
                                        <td><?php echo $consumo['cantidad']; ?></td>
                                        <td><?php echo formatMoney($consumo['precio_unitario']); ?></td>
                                        <td><strong><?php echo formatMoney($consumo['subtotal']); ?></strong></td>
                                        <td>
                                            <a href="?reserva_id=<?php echo $reserva_id; ?>&eliminar_consumo=<?php echo $consumo['id']; ?>" 
                                               class="btn btn-sm btn-danger"
                                               onclick="return confirm('¿Eliminar este consumo?')">🗑️</a>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>

    <footer class="footer">
        <div class="container">
            <p>&copy; <?php echo date('Y'); ?> Sistema de Reservas para Posadas.</p>
        </div>
    </footer>

    <script src="../assets/js/main.js"></script>
</body>
</html>
