<?php
/**
 * Configuración General del Sistema
 * Sistema de Administración de Veterinaria
 */

// Configuración de sesión
ini_set('session.cookie_httponly', 1);
ini_set('session.use_only_cookies', 1);
ini_set('session.cookie_secure', 0); // Cambiar a 1 si se usa HTTPS
session_start();

// Zona horaria
date_default_timezone_set('America/Asuncion');

// Configuración de errores (cambiar en producción)
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Rutas del sistema
define('BASE_PATH', dirname(__DIR__));
define('CONFIG_PATH', BASE_PATH . '/config');
define('INCLUDES_PATH', BASE_PATH . '/includes');
define('UPLOADS_PATH', BASE_PATH . '/uploads');
define('ASSETS_PATH', BASE_PATH . '/assets');

// URLs del sistema
$protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
$host = $_SERVER['HTTP_HOST'];
// Obtener la ruta base desde la carpeta raíz del proyecto
$script_path = str_replace('\\', '/', dirname($_SERVER['SCRIPT_NAME']));
// Eliminar /admin, /colaborador, /paciente, /config, /includes si están presentes
$script_path = preg_replace('#/(admin|colaborador|paciente|config|includes)$#', '', $script_path);
$base_url = $protocol . '://' . $host . $script_path;
define('BASE_URL', rtrim($base_url, '/'));

// Configuración de la aplicación
define('APP_NAME', 'Sistema de Administración Veterinaria');
define('APP_VERSION', '1.4.1');
define('MONEDA', 'Gs.');
define('MONEDA_NOMBRE', 'Guaraníes');

// Configuración de archivos
define('MAX_FILE_SIZE', 5 * 1024 * 1024); // 5MB
define('ALLOWED_IMAGE_TYPES', ['image/jpeg', 'image/png', 'image/gif']);

// Incluir archivo de base de datos
require_once CONFIG_PATH . '/database.php';

/**
 * Función para formatear números con puntos de miles
 */
function formatear_numero($numero) {
    return number_format($numero, 0, ',', '.');
}

/**
 * Función para formatear moneda en guaraníes
 */
function formatear_moneda($monto) {
    if ($monto === null || $monto === '') {
        return MONEDA . ' 0';
    }
    return MONEDA . ' ' . formatear_numero($monto);
}

/**
 * Función para formatear fecha
 */
function formatear_fecha($fecha, $incluir_hora = false) {
    if (empty($fecha)) return '';
    
    $timestamp = strtotime($fecha);
    if ($incluir_hora) {
        return date('d/m/Y H:i', $timestamp);
    }
    return date('d/m/Y', $timestamp);
}

/**
 * Función para sanitizar entrada de datos
 */
function limpiar_entrada($data) {
    $data = trim($data);
    $data = stripslashes($data);
    $data = htmlspecialchars($data, ENT_QUOTES, 'UTF-8');
    return $data;
}

/**
 * Función para verificar si el usuario está autenticado
 */
function esta_autenticado() {
    return isset($_SESSION['usuario_id']) && !empty($_SESSION['usuario_id']);
}

/**
 * Función para verificar el rol del usuario
 */
function tiene_rol($rol) {
    return isset($_SESSION['rol']) && $_SESSION['rol'] === $rol;
}

/**
 * Función para requerir autenticación
 */
function requerir_autenticacion() {
    if (!esta_autenticado()) {
        header('Location: ' . BASE_URL . '/login.php');
        exit();
    }
}

/**
 * Función para requerir rol específico
 */
function requerir_rol($rol) {
    requerir_autenticacion();
    if (!tiene_rol($rol)) {
        // Redirigir al dashboard correcto según el rol del usuario
        $rol_actual = $_SESSION['rol'] ?? '';
        if ($rol_actual === 'administrador') {
            header('Location: ' . BASE_URL . '/admin/index.php');
        } elseif ($rol_actual === 'colaborador') {
            header('Location: ' . BASE_URL . '/colaborador/index.php');
        } elseif ($rol_actual === 'paciente') {
            header('Location: ' . BASE_URL . '/paciente/index.php');
        } else {
            header('Location: ' . BASE_URL . '/login.php');
        }
        exit();
    }
}

/**
 * Función para generar código de barras único
 */
function generar_codigo_barras() {
    return 'VET' . date('Ymd') . rand(1000, 9999);
}

/**
 * Función para generar número de factura
 */
function generar_numero_factura() {
    return 'FAC-' . date('Ymd') . '-' . rand(1000, 9999);
}

/**
 * Función para crear enlace de WhatsApp
 */
function crear_enlace_whatsapp($telefono) {
    // Limpiar el número de teléfono
    $telefono_limpio = preg_replace('/[^0-9]/', '', $telefono);
    
    // Si no empieza con 595, agregarlo
    if (substr($telefono_limpio, 0, 3) !== '595') {
        $telefono_limpio = '595' . $telefono_limpio;
    }
    
    return 'https://wa.me/' . $telefono_limpio;
}

/**
 * Función para crear enlace de Telegram
 */
function crear_enlace_telegram($telefono) {
    // Limpiar el número de teléfono
    $telefono_limpio = preg_replace('/[^0-9]/', '', $telefono);
    
    // Si no empieza con 595, agregarlo
    if (substr($telefono_limpio, 0, 3) !== '595') {
        $telefono_limpio = '595' . $telefono_limpio;
    }
    
    return 'https://t.me/' . $telefono_limpio;
}

/**
 * Función para subir archivo
 */
function subir_archivo($archivo, $carpeta = 'general') {
    if (!isset($archivo) || $archivo['error'] !== UPLOAD_ERR_OK) {
        return ['success' => false, 'message' => 'Error al subir el archivo'];
    }
    
    // Verificar tamaño
    if ($archivo['size'] > MAX_FILE_SIZE) {
        return ['success' => false, 'message' => 'El archivo es demasiado grande'];
    }
    
    // Verificar tipo
    if (!in_array($archivo['type'], ALLOWED_IMAGE_TYPES)) {
        return ['success' => false, 'message' => 'Tipo de archivo no permitido'];
    }
    
    // Crear carpeta si no existe
    $ruta_carpeta = UPLOADS_PATH . '/' . $carpeta;
    if (!file_exists($ruta_carpeta)) {
        mkdir($ruta_carpeta, 0755, true);
    }
    
    // Generar nombre único
    $extension = pathinfo($archivo['name'], PATHINFO_EXTENSION);
    $nombre_archivo = uniqid() . '_' . time() . '.' . $extension;
    $ruta_completa = $ruta_carpeta . '/' . $nombre_archivo;
    
    // Mover archivo
    if (move_uploaded_file($archivo['tmp_name'], $ruta_completa)) {
        return [
            'success' => true, 
            'filename' => $nombre_archivo,
            'path' => $nombre_archivo  // Solo el nombre del archivo, sin la carpeta
        ];
    }
    
    return ['success' => false, 'message' => 'Error al guardar el archivo'];
}

/**
 * Función para redirigir con mensaje
 */
function redirigir_con_mensaje($url, $mensaje, $tipo = 'success') {
    $_SESSION['mensaje'] = $mensaje;
    $_SESSION['mensaje_tipo'] = $tipo;
    header('Location: ' . $url);
    exit();
}

/**
 * Función para mostrar mensaje de sesión
 */
function mostrar_mensaje() {
    if (isset($_SESSION['mensaje'])) {
        $tipo = isset($_SESSION['mensaje_tipo']) ? $_SESSION['mensaje_tipo'] : 'info';
        $clase_alerta = 'alert-' . $tipo;
        
        echo '<div class="alert ' . $clase_alerta . ' alert-dismissible fade show" role="alert">';
        echo htmlspecialchars($_SESSION['mensaje']);
        echo '<button type="button" class="btn-close" data-bs-dismiss="alert"></button>';
        echo '</div>';
        
        unset($_SESSION['mensaje']);
        unset($_SESSION['mensaje_tipo']);
    }
}
