<?php
/**
 * Ver Detalles de Mascota
 * Sistema de Administración de Veterinaria
 */

require_once '../config/config.php';
requerir_autenticacion();

if (!tiene_rol('colaborador') && !tiene_rol('administrador')) {
    header('Location: ' . BASE_URL . '/index.php');
    exit();
}

$mascota_id = (int)($_GET['id'] ?? 0);

if ($mascota_id <= 0) {
    redirigir_con_mensaje('mascotas.php', 'Mascota no encontrada', 'danger');
}

try {
    $db = Database::getInstance()->getConnection();
    
    // Obtener datos de la mascota y dueño
    $stmt = $db->prepare("SELECT m.*, d.* 
                          FROM mascotas m
                          INNER JOIN duenos d ON m.dueno_id = d.id
                          WHERE m.id = :id");
    $stmt->execute(['id' => $mascota_id]);
    $mascota = $stmt->fetch();
    
    if (!$mascota) {
        redirigir_con_mensaje('mascotas.php', 'Mascota no encontrada', 'danger');
    }
    
    // Obtener historial de visitas
    $stmt = $db->prepare("SELECT v.*, u.nombre_completo as profesional_nombre, f.total as factura_total, f.estado as factura_estado
                          FROM visitas v
                          INNER JOIN usuarios u ON v.profesional_id = u.id
                          LEFT JOIN facturas f ON v.id = f.visita_id
                          WHERE v.mascota_id = :mascota_id
                          ORDER BY v.fecha_visita DESC");
    $stmt->execute(['mascota_id' => $mascota_id]);
    $visitas = $stmt->fetchAll();
    
    // Obtener vacunas
    $stmt = $db->prepare("SELECT v.*, u.nombre_completo as profesional_nombre
                          FROM vacunas v
                          LEFT JOIN usuarios u ON v.profesional_id = u.id
                          WHERE v.mascota_id = :mascota_id
                          ORDER BY v.fecha_aplicacion DESC");
    $stmt->execute(['mascota_id' => $mascota_id]);
    $vacunas = $stmt->fetchAll();
    
    // Calcular edad
    $edad = '';
    if ($mascota['fecha_nacimiento']) {
        $fecha_nac = new DateTime($mascota['fecha_nacimiento']);
        $hoy = new DateTime();
        $diff = $hoy->diff($fecha_nac);
        if ($diff->y > 0) {
            $edad = $diff->y . ' año' . ($diff->y > 1 ? 's' : '');
        } else if ($diff->m > 0) {
            $edad = $diff->m . ' mes' . ($diff->m > 1 ? 'es' : '');
        } else {
            $edad = $diff->d . ' día' . ($diff->d > 1 ? 's' : '');
        }
    }
    
} catch (PDOException $e) {
    die("Error en la base de datos: " . $e->getMessage());
}

$titulo_pagina = 'Ficha de ' . $mascota['nombre'];

include '../includes/header.php';
?>

<div class="row mb-4">
    <div class="col-md-8">
        <h1><i class="bi bi-heart"></i> Ficha de <?php echo htmlspecialchars($mascota['nombre']); ?></h1>
    </div>
    <div class="col-md-4 text-end">
        <a href="historial_clinico.php?id=<?php echo $mascota_id; ?>" class="btn btn-primary">
            <i class="bi bi-clipboard2-pulse"></i> Historial Clínico
        </a>
        <a href="editar_mascota.php?id=<?php echo $mascota_id; ?>" class="btn btn-warning">
            <i class="bi bi-pencil"></i> Editar
        </a>
        <a href="nueva_visita.php?mascota_id=<?php echo $mascota_id; ?>" class="btn btn-success">
            <i class="bi bi-plus-circle"></i> Nueva Visita
        </a>
        <a href="generar_credencial.php?id=<?php echo $mascota_id; ?>" class="btn btn-info" target="_blank">
            <i class="bi bi-card-heading"></i> Ver Credencial
        </a>
    </div>
</div>

<div class="row">
    <!-- Información de la Mascota -->
    <div class="col-md-4">
        <div class="card mb-4">
            <div class="card-header bg-primary text-white">
                <h5 class="mb-0"><i class="bi bi-heart"></i> Información de la Mascota</h5>
            </div>
            <div class="card-body text-center">
                <?php if ($mascota['foto']): ?>
                    <img src="<?php echo BASE_URL; ?>/uploads/mascotas/<?php echo htmlspecialchars($mascota['foto']); ?>" 
                         class="img-fluid rounded mb-3" alt="Foto de <?php echo htmlspecialchars($mascota['nombre']); ?>"
                         style="max-height: 250px;">
                <?php else: ?>
                    <i class="bi bi-heart-fill text-muted mb-3" style="font-size: 6rem;"></i>
                <?php endif; ?>
                
                <h3><?php echo htmlspecialchars($mascota['nombre']); ?></h3>
                <p class="mb-2"><strong>Código:</strong> <code><?php echo htmlspecialchars($mascota['codigo_barras']); ?></code></p>
                <p class="mb-2"><strong>Tipo:</strong> <?php echo ucfirst($mascota['tipo_mascota']); ?></p>
                <?php if ($mascota['raza']): ?>
                    <p class="mb-2"><strong>Raza:</strong> <?php echo htmlspecialchars($mascota['raza']); ?></p>
                <?php endif; ?>
                <?php if ($edad): ?>
                    <p class="mb-2"><strong>Edad:</strong> <?php echo $edad; ?></p>
                <?php endif; ?>
                <?php if ($mascota['peso']): ?>
                    <p class="mb-2"><strong>Peso:</strong> <?php echo $mascota['peso']; ?> kg</p>
                <?php endif; ?>
                <p class="mb-2">
                    <strong>Estado:</strong> 
                    <?php if ($mascota['activo']): ?>
                        <span class="badge bg-success">Activo</span>
                    <?php else: ?>
                        <span class="badge bg-secondary">Inactivo</span>
                    <?php endif; ?>
                </p>
            </div>
        </div>
        
        <!-- Información del Dueño -->
        <div class="card mb-4">
            <div class="card-header bg-info text-white">
                <h5 class="mb-0"><i class="bi bi-person"></i> Información del Dueño</h5>
            </div>
            <div class="card-body">
                <p class="mb-2"><strong><?php echo htmlspecialchars($mascota['nombre_completo']); ?></strong></p>
                <?php if ($mascota['cedula']): ?>
                    <p class="mb-2"><i class="bi bi-card-text"></i> CI: <?php echo htmlspecialchars($mascota['cedula']); ?></p>
                <?php endif; ?>
                <p class="mb-2">
                    <i class="bi bi-telephone"></i> 
                    <a href="<?php echo crear_enlace_whatsapp($mascota['telefono']); ?>" target="_blank" class="text-decoration-none">
                        <?php echo htmlspecialchars($mascota['telefono']); ?>
                    </a>
                </p>
                <div class="mb-2">
                    <?php if ($mascota['tiene_whatsapp']): ?>
                        <a href="<?php echo crear_enlace_whatsapp($mascota['telefono']); ?>" 
                           target="_blank" class="btn btn-sm btn-success">
                            <i class="bi bi-whatsapp"></i> WhatsApp
                        </a>
                    <?php endif; ?>
                    <?php if ($mascota['tiene_telegram']): ?>
                        <a href="<?php echo crear_enlace_telegram($mascota['telefono']); ?>" 
                           target="_blank" class="btn btn-sm btn-info">
                            <i class="bi bi-telegram"></i> Telegram
                        </a>
                    <?php endif; ?>
                </div>
                <?php if ($mascota['email']): ?>
                    <p class="mb-2"><i class="bi bi-envelope"></i> <?php echo htmlspecialchars($mascota['email']); ?></p>
                <?php endif; ?>
                <?php if ($mascota['direccion']): ?>
                    <p class="mb-0"><i class="bi bi-geo-alt"></i> <?php echo htmlspecialchars($mascota['direccion']); ?></p>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <!-- Historial y Vacunas -->
    <div class="col-md-8">
        <!-- Tabs -->
        <ul class="nav nav-tabs mb-3" role="tablist">
            <li class="nav-item">
                <a class="nav-link active" data-bs-toggle="tab" href="#tabVisitas">
                    <i class="bi bi-calendar-check"></i> Historial de Visitas (<?php echo count($visitas); ?>)
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link" data-bs-toggle="tab" href="#tabVacunas">
                    <i class="bi bi-shield-check"></i> Vacunas (<?php echo count($vacunas); ?>)
                </a>
            </li>
        </ul>
        
        <div class="tab-content">
            <!-- Tab Visitas -->
            <div class="tab-pane fade show active" id="tabVisitas">
                <?php if (count($visitas) > 0): ?>
                    <?php foreach ($visitas as $visita): ?>
                        <div class="card mb-3">
                            <div class="card-header">
                                <div class="d-flex justify-content-between align-items-center">
                                    <div>
                                        <strong><?php echo formatear_fecha($visita['fecha_visita'], true); ?></strong>
                                        <span class="text-muted">- Dr. <?php echo htmlspecialchars($visita['profesional_nombre']); ?></span>
                                    </div>
                                    <div>
                                        <?php if ($visita['factura_total']): ?>
                                            <span class="badge bg-<?php echo $visita['factura_estado'] === 'pagada' ? 'success' : 'warning'; ?>">
                                                <?php echo formatear_moneda($visita['factura_total']); ?>
                                            </span>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                            <div class="card-body">
                                <?php if ($visita['motivo']): ?>
                                    <p class="mb-2"><strong>Motivo:</strong> <?php echo htmlspecialchars($visita['motivo']); ?></p>
                                <?php endif; ?>
                                <?php if ($visita['diagnostico']): ?>
                                    <p class="mb-2"><strong>Diagnóstico:</strong> <?php echo htmlspecialchars($visita['diagnostico']); ?></p>
                                <?php endif; ?>
                                <?php if ($visita['observaciones']): ?>
                                    <p class="mb-2"><strong>Observaciones:</strong> <?php echo htmlspecialchars($visita['observaciones']); ?></p>
                                <?php endif; ?>
                                <?php if ($visita['peso_actual']): ?>
                                    <p class="mb-2"><strong>Peso:</strong> <?php echo $visita['peso_actual']; ?> kg</p>
                                <?php endif; ?>
                                <?php if ($visita['proxima_cita']): ?>
                                    <p class="mb-0">
                                        <i class="bi bi-calendar-event text-primary"></i> 
                                        <strong>Próxima cita:</strong> <?php echo formatear_fecha($visita['proxima_cita']); ?>
                                    </p>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php else: ?>
                    <div class="alert alert-info">
                        <i class="bi bi-info-circle"></i> No hay visitas registradas para esta mascota
                    </div>
                <?php endif; ?>
            </div>
            
            <!-- Tab Vacunas -->
            <div class="tab-pane fade" id="tabVacunas">
                <?php if (count($vacunas) > 0): ?>
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th>Vacuna</th>
                                    <th>Fecha Aplicación</th>
                                    <th>Próxima Dosis</th>
                                    <th>Profesional</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($vacunas as $vacuna): ?>
                                    <tr>
                                        <td><strong><?php echo htmlspecialchars($vacuna['nombre_vacuna']); ?></strong></td>
                                        <td><?php echo formatear_fecha($vacuna['fecha_aplicacion']); ?></td>
                                        <td>
                                            <?php if ($vacuna['proxima_dosis']): ?>
                                                <?php
                                                $proxima = new DateTime($vacuna['proxima_dosis']);
                                                $hoy = new DateTime();
                                                $vencida = $proxima < $hoy;
                                                $proxima_15 = $proxima->diff($hoy)->days <= 15 && !$vencida;
                                                ?>
                                                <span class="badge bg-<?php echo $vencida ? 'danger' : ($proxima_15 ? 'warning' : 'success'); ?>">
                                                    <?php echo formatear_fecha($vacuna['proxima_dosis']); ?>
                                                </span>
                                            <?php else: ?>
                                                -
                                            <?php endif; ?>
                                        </td>
                                        <td><?php echo htmlspecialchars($vacuna['profesional_nombre'] ?? '-'); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <div class="alert alert-info">
                        <i class="bi bi-info-circle"></i> No hay vacunas registradas para esta mascota
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<?php include '../includes/footer.php'; ?>
