<?php
/**
 * Registrar Nueva Mascota
 * Sistema de Administración de Veterinaria
 */

require_once '../config/config.php';
requerir_autenticacion();

if (!tiene_rol('colaborador') && !tiene_rol('administrador')) {
    header('Location: ' . BASE_URL . '/index.php');
    exit();
}

$titulo_pagina = 'Registrar Nueva Mascota';

// Procesar formulario
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $db = Database::getInstance()->getConnection();
        $db->beginTransaction();
        
        // Verificar si se seleccionó un dueño existente
        $dueno_id_existente = !empty($_POST['dueno_id_existente']) ? (int)$_POST['dueno_id_existente'] : 0;
        
        if ($dueno_id_existente > 0) {
            // Usar dueño existente seleccionado
            $dueno_id = $dueno_id_existente;
        } else {
            // Datos del dueño nuevo
            $dueno_nombre = limpiar_entrada($_POST['dueno_nombre']);
            $dueno_cedula = limpiar_entrada($_POST['dueno_cedula']);
            $dueno_direccion = limpiar_entrada($_POST['dueno_direccion']);
            $dueno_telefono = limpiar_entrada($_POST['dueno_telefono']);
            $tiene_whatsapp = isset($_POST['tiene_whatsapp']) ? 1 : 0;
            $tiene_telegram = isset($_POST['tiene_telegram']) ? 1 : 0;
            $dueno_email = limpiar_entrada($_POST['dueno_email']);
            
            // Verificar si el dueño ya existe por teléfono o cédula
            $stmt = $db->prepare("SELECT id FROM duenos WHERE telefono = :telefono OR cedula = :cedula LIMIT 1");
            $stmt->execute(['telefono' => $dueno_telefono, 'cedula' => $dueno_cedula]);
            $dueno_existente = $stmt->fetch();
            
            if ($dueno_existente) {
                // Dueño ya existe, usar ese ID (permite múltiples mascotas por dueño)
                $dueno_id = $dueno_existente['id'];
            } else {
            // Crear nuevo dueño
            $usuario_paciente = strtolower(str_replace(' ', '', $dueno_nombre)) . rand(100, 999);
            $contrasena_paciente = 'mascota' . rand(1000, 9999);
            $contrasena_hash = password_hash($contrasena_paciente, PASSWORD_DEFAULT);
            
            $stmt = $db->prepare("INSERT INTO duenos (nombre_completo, cedula, direccion, telefono, 
                                  tiene_whatsapp, tiene_telegram, email, usuario_paciente, contrasena_paciente) 
                                  VALUES (:nombre, :cedula, :direccion, :telefono, :whatsapp, :telegram, :email, :usuario, :contrasena)");
            $stmt->execute([
                'nombre' => $dueno_nombre,
                'cedula' => $dueno_cedula,
                'direccion' => $dueno_direccion,
                'telefono' => $dueno_telefono,
                'whatsapp' => $tiene_whatsapp,
                'telegram' => $tiene_telegram,
                'email' => $dueno_email,
                'usuario' => $usuario_paciente,
                'contrasena' => $contrasena_hash
            ]);
            $dueno_id = $db->lastInsertId();
            }
        }
        
        // Datos de la mascota
        $nombre_mascota = limpiar_entrada($_POST['nombre_mascota']);
        $tipo_mascota = limpiar_entrada($_POST['tipo_mascota']);
        $raza = limpiar_entrada($_POST['raza']);
        $fecha_nacimiento = $_POST['fecha_nacimiento'];
        $peso = !empty($_POST['peso']) ? (float)$_POST['peso'] : null;
        
        // Generar código de barras único
        $codigo_barras = generar_codigo_barras();
        while (true) {
            $stmt = $db->prepare("SELECT id FROM mascotas WHERE codigo_barras = :codigo");
            $stmt->execute(['codigo' => $codigo_barras]);
            if (!$stmt->fetch()) break;
            $codigo_barras = generar_codigo_barras();
        }
        
        // Procesar foto
        $foto_path = null;
        if (isset($_FILES['foto']) && $_FILES['foto']['error'] === UPLOAD_ERR_OK) {
            $resultado = subir_archivo($_FILES['foto'], 'mascotas');
            if ($resultado['success']) {
                $foto_path = $resultado['path'];
            }
        }
        
        // Insertar mascota
        $stmt = $db->prepare("INSERT INTO mascotas (dueno_id, nombre, tipo_mascota, raza, fecha_nacimiento, peso, foto, codigo_barras) 
                              VALUES (:dueno_id, :nombre, :tipo, :raza, :fecha_nac, :peso, :foto, :codigo)");
        $stmt->execute([
            'dueno_id' => $dueno_id,
            'nombre' => $nombre_mascota,
            'tipo' => $tipo_mascota,
            'raza' => $raza,
            'fecha_nac' => $fecha_nacimiento,
            'peso' => $peso,
            'foto' => $foto_path,
            'codigo' => $codigo_barras
        ]);
        $mascota_id = $db->lastInsertId();
        
        $db->commit();
        
        $mensaje = 'Mascota registrada exitosamente';
        if ($dueno_existente) {
            $mensaje .= '. Se asoció con el dueño existente (ahora tiene ' . 
                       ($db->query("SELECT COUNT(*) FROM mascotas WHERE dueno_id = $dueno_id")->fetchColumn()) . ' mascota(s))';
        }
        
        redirigir_con_mensaje('ver_mascota.php?id=' . $mascota_id, $mensaje, 'success');
        
    } catch (PDOException $e) {
        $db->rollBack();
        $error = "Error al registrar la mascota: " . $e->getMessage();
    }
}

include '../includes/header.php';
?>

<div class="row mb-4">
    <div class="col-12">
        <h1><i class="bi bi-plus-circle"></i> Registrar Nueva Mascota</h1>
    </div>
</div>

<?php if (isset($error)): ?>
    <div class="alert alert-danger alert-dismissible fade show">
        <i class="bi bi-exclamation-triangle"></i> <?php echo htmlspecialchars($error); ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<form method="POST" action="" enctype="multipart/form-data">
    <div class="row">
        <!-- Datos del Dueño -->
        <div class="col-md-6">
            <div class="card mb-4">
                <div class="card-header bg-primary text-white">
                    <h5 class="mb-0"><i class="bi bi-person"></i> Datos del Dueño</h5>
                </div>
                <div class="card-body">
                    <div class="alert alert-success">
                        <i class="bi bi-search"></i> <strong>¿El dueño ya existe?</strong> Busque por nombre, cédula o teléfono y selecciónelo.
                    </div>
                    
                    <!-- Buscador de Dueños Existentes -->
                    <div class="mb-3">
                        <label class="form-label"><i class="bi bi-search"></i> Buscar Dueño Existente</label>
                        <input type="text" class="form-control" id="buscarDueno" 
                               placeholder="Escriba nombre, cédula o teléfono..." autocomplete="off">
                        <div id="resultadosDuenos" class="list-group mt-2" style="display:none;"></div>
                        <input type="hidden" id="dueno_id_existente" name="dueno_id_existente" value="">
                    </div>
                    
                    <div class="text-center mb-3">
                        <button type="button" class="btn btn-sm btn-outline-primary" id="btnNuevoDueno">
                            <i class="bi bi-plus-circle"></i> O registrar nuevo dueño
                        </button>
                    </div>
                    
                    <hr>
                    
                    <!-- Formulario de Dueño (oculto inicialmente si se busca) -->
                    <div id="formularioDueno">
                        <div class="mb-3">
                            <label class="form-label">Nombre Completo *</label>
                            <input type="text" class="form-control" name="dueno_nombre" id="dueno_nombre" required>
                    </div>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Cédula de Identidad</label>
                            <input type="text" class="form-control" name="dueno_cedula" id="dueno_cedula" placeholder="1234567">
                    </div>
                        <div class="mb-3">
                            <label class="form-label">Dirección</label>
                            <textarea class="form-control" name="dueno_direccion" id="dueno_direccion" rows="2"></textarea>
                    </div>
                        <div class="mb-3">
                            <label class="form-label">Teléfono *</label>
                            <input type="text" class="form-control" name="dueno_telefono" id="dueno_telefono" 
                                   placeholder="+595..." required>
                        <small class="text-muted">Formato: +595 + número</small>
                    </div>
                    <div class="mb-3">
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" name="tiene_whatsapp" id="tiene_whatsapp">
                            <label class="form-check-label" for="tiene_whatsapp">
                                <i class="bi bi-whatsapp text-success"></i> Tiene WhatsApp
                            </label>
                        </div>
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" name="tiene_telegram" id="tiene_telegram">
                            <label class="form-check-label" for="tiene_telegram">
                                <i class="bi bi-telegram text-info"></i> Tiene Telegram
                            </label>
                        </div>
                    </div>
                        <div class="mb-3">
                            <label class="form-label">Email</label>
                            <input type="email" class="form-control" name="dueno_email" id="dueno_email">
                        </div>
                    </div>
                    <!-- Fin formularioDueno -->
                </div>
            </div>
        </div>
        
        <!-- Datos de la Mascota -->
        <div class="col-md-6">
            <div class="card mb-4">
                <div class="card-header bg-success text-white">
                    <h5 class="mb-0"><i class="bi bi-heart"></i> Datos de la Mascota</h5>
                </div>
                <div class="card-body">
                    <div class="mb-3">
                        <label class="form-label">Nombre de la Mascota *</label>
                        <input type="text" class="form-control" name="nombre_mascota" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Tipo de Mascota *</label>
                        <select class="form-select" name="tipo_mascota" required>
                            <option value="">Seleccione...</option>
                            <option value="perro">Perro</option>
                            <option value="gato">Gato</option>
                            <option value="ave">Ave</option>
                            <option value="conejo">Conejo</option>
                            <option value="hamster">Hámster</option>
                            <option value="reptil">Reptil</option>
                            <option value="otro">Otro</option>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Raza</label>
                        <input type="text" class="form-control" name="raza">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Fecha de Nacimiento</label>
                        <input type="date" class="form-control" name="fecha_nacimiento">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Peso (kg) - Opcional</label>
                        <input type="number" class="form-control" name="peso" step="0.01" placeholder="5.5">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Fotografía</label>
                        <input type="file" class="form-control" name="foto" accept="image/*">
                        <small class="text-muted">Formatos permitidos: JPG, PNG, GIF. Máximo 5MB</small>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <div class="card">
        <div class="card-body text-center">
            <button type="submit" class="btn btn-primary btn-lg">
                <i class="bi bi-save"></i> Registrar Mascota
            </button>
            <a href="index.php" class="btn btn-secondary btn-lg">
                <i class="bi bi-x-circle"></i> Cancelar
            </a>
        </div>
    </div>
</form>

<?php include '../includes/footer.php'; ?>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const buscarDueno = document.getElementById('buscarDueno');
    const resultadosDuenos = document.getElementById('resultadosDuenos');
    const formularioDueno = document.getElementById('formularioDueno');
    const btnNuevoDueno = document.getElementById('btnNuevoDueno');
    const duenoIdExistente = document.getElementById('dueno_id_existente');
    
    let timeoutBusqueda = null;
    let duenoSeleccionado = false;
    
    // Buscar dueños mientras escribe
    buscarDueno.addEventListener('input', function() {
        const termino = this.value.trim();
        
        if (termino.length < 2) {
            resultadosDuenos.style.display = 'none';
            resultadosDuenos.innerHTML = '';
            return;
        }
        
        clearTimeout(timeoutBusqueda);
        timeoutBusqueda = setTimeout(function() {
            buscarDuenosExistentes(termino);
        }, 300);
    });
    
    // Función para buscar dueños
    function buscarDuenosExistentes(termino) {
        const baseUrl = document.querySelector('meta[name="base-url"]')?.content || '';
        
        fetch(`api_buscar_duenos.php?q=${encodeURIComponent(termino)}`)
            .then(response => response.json())
            .then(data => {
                if (data.error) {
                    console.error(data.error);
                    return;
                }
                
                if (data.length === 0) {
                    resultadosDuenos.innerHTML = '<div class="list-group-item text-muted">No se encontraron dueños</div>';
                    resultadosDuenos.style.display = 'block';
                    return;
                }
                
                let html = '';
                data.forEach(dueno => {
                    html += `
                        <a href="#" class="list-group-item list-group-item-action" data-dueno='${JSON.stringify(dueno)}'>
                            <div class="d-flex w-100 justify-content-between">
                                <h6 class="mb-1">${dueno.nombre_completo}</h6>
                                <small class="text-success">${dueno.total_mascotas} mascota(s)</small>
                            </div>
                            <p class="mb-1">
                                <small><i class="bi bi-credit-card"></i> ${dueno.cedula}</small> | 
                                <small><i class="bi bi-telephone"></i> ${dueno.telefono}</small>
                            </p>
                        </a>
                    `;
                });
                
                resultadosDuenos.innerHTML = html;
                resultadosDuenos.style.display = 'block';
                
                // Agregar eventos de clic a los resultados
                document.querySelectorAll('#resultadosDuenos .list-group-item').forEach(item => {
                    item.addEventListener('click', function(e) {
                        e.preventDefault();
                        const dueno = JSON.parse(this.dataset.dueno);
                        seleccionarDueno(dueno);
                    });
                });
            })
            .catch(error => {
                console.error('Error en la búsqueda:', error);
            });
    }
    
    // Función para seleccionar un dueño
    function seleccionarDueno(dueno) {
        duenoSeleccionado = true;
        duenoIdExistente.value = dueno.id;
        
        // Llenar los campos con los datos del dueño
        document.getElementById('dueno_nombre').value = dueno.nombre_completo;
        document.getElementById('dueno_cedula').value = dueno.cedula;
        document.getElementById('dueno_telefono').value = dueno.telefono;
        document.getElementById('dueno_email').value = dueno.email || '';
        document.getElementById('dueno_direccion').value = dueno.direccion || '';
        document.getElementById('tiene_whatsapp').checked = dueno.tiene_whatsapp;
        document.getElementById('tiene_telegram').checked = dueno.tiene_telegram;
        
        // Deshabilitar los campos del formulario
        document.getElementById('dueno_nombre').disabled = true;
        document.getElementById('dueno_cedula').disabled = true;
        document.getElementById('dueno_telefono').disabled = true;
        document.getElementById('dueno_email').disabled = true;
        document.getElementById('dueno_direccion').disabled = true;
        document.getElementById('tiene_whatsapp').disabled = true;
        document.getElementById('tiene_telegram').disabled = true;
        
        // Mostrar mensaje de éxito
        buscarDueno.value = `✓ ${dueno.nombre_completo} - ${dueno.cedula}`;
        buscarDueno.classList.add('is-valid');
        buscarDueno.disabled = true;
        
        resultadosDuenos.style.display = 'none';
        
        // Mostrar botón para cambiar de dueño
        btnNuevoDueno.innerHTML = '<i class="bi bi-x-circle"></i> Cambiar dueño';
        btnNuevoDueno.classList.remove('btn-outline-primary');
        btnNuevoDueno.classList.add('btn-outline-danger');
    }
    
    // Botón para nuevo dueño o cambiar dueño
    btnNuevoDueno.addEventListener('click', function() {
        if (duenoSeleccionado) {
            // Limpiar selección
            duenoSeleccionado = false;
            duenoIdExistente.value = '';
            
            // Habilitar campos
            document.getElementById('dueno_nombre').disabled = false;
            document.getElementById('dueno_cedula').disabled = false;
            document.getElementById('dueno_telefono').disabled = false;
            document.getElementById('dueno_email').disabled = false;
            document.getElementById('dueno_direccion').disabled = false;
            document.getElementById('tiene_whatsapp').disabled = false;
            document.getElementById('tiene_telegram').disabled = false;
            
            // Limpiar campos
            document.getElementById('dueno_nombre').value = '';
            document.getElementById('dueno_cedula').value = '';
            document.getElementById('dueno_telefono').value = '';
            document.getElementById('dueno_email').value = '';
            document.getElementById('dueno_direccion').value = '';
            document.getElementById('tiene_whatsapp').checked = false;
            document.getElementById('tiene_telegram').checked = false;
            
            // Limpiar búsqueda
            buscarDueno.value = '';
            buscarDueno.classList.remove('is-valid');
            buscarDueno.disabled = false;
            
            // Restaurar botón
            btnNuevoDueno.innerHTML = '<i class="bi bi-plus-circle"></i> O registrar nuevo dueño';
            btnNuevoDueno.classList.remove('btn-outline-danger');
            btnNuevoDueno.classList.add('btn-outline-primary');
        }
    });
    
    // Ocultar resultados al hacer clic fuera
    document.addEventListener('click', function(e) {
        if (!buscarDueno.contains(e.target) && !resultadosDuenos.contains(e.target)) {
            resultadosDuenos.style.display = 'none';
        }
    });
});
</script>
