<?php
/**
 * Historial Clínico de Mascota
 * Sistema de Administración de Veterinaria
 */

require_once '../config/config.php';
requerir_autenticacion();

if (!tiene_rol('colaborador') && !tiene_rol('administrador')) {
    header('Location: ' . BASE_URL . '/index.php');
    exit();
}

$titulo_pagina = 'Historial Clínico';

$mascota_id = (int)($_GET['id'] ?? 0);

if ($mascota_id <= 0) {
    redirigir_con_mensaje('mascotas.php', 'ID de mascota inválido', 'error');
}

try {
    $db = Database::getInstance()->getConnection();
    
    // Obtener datos de la mascota
    $stmt = $db->prepare("SELECT m.*, d.nombre_completo as dueno_nombre, d.telefono, d.cedula, d.email
                          FROM mascotas m 
                          INNER JOIN duenos d ON m.dueno_id = d.id 
                          WHERE m.id = :id");
    $stmt->execute(['id' => $mascota_id]);
    $mascota = $stmt->fetch();
    
    if (!$mascota) {
        redirigir_con_mensaje('mascotas.php', 'Mascota no encontrada', 'error');
    }
    
    // Obtener todas las visitas con datos del profesional
    $stmt = $db->prepare("SELECT v.*, u.nombre as profesional_nombre
                          FROM visitas v
                          LEFT JOIN usuarios u ON v.profesional_id = u.id
                          WHERE v.mascota_id = :id
                          ORDER BY v.fecha_visita DESC");
    $stmt->execute(['id' => $mascota_id]);
    $visitas = $stmt->fetchAll();
    
    // Obtener todas las vacunas
    $stmt = $db->prepare("SELECT vac.*, u.nombre as profesional_nombre
                          FROM vacunas vac
                          LEFT JOIN usuarios u ON vac.profesional_id = u.id
                          WHERE vac.mascota_id = :id
                          ORDER BY vac.fecha_aplicacion DESC");
    $stmt->execute(['id' => $mascota_id]);
    $vacunas = $stmt->fetchAll();
    
    // Obtener evolución del peso
    $stmt = $db->prepare("SELECT fecha_visita, peso_actual 
                          FROM visitas 
                          WHERE mascota_id = :id AND peso_actual IS NOT NULL
                          ORDER BY fecha_visita ASC");
    $stmt->execute(['id' => $mascota_id]);
    $pesos = $stmt->fetchAll();
    
    // Calcular estadísticas
    $total_visitas = count($visitas);
    $total_vacunas = count($vacunas);
    
    // Última visita
    $ultima_visita = $visitas[0] ?? null;
    
    // Próxima cita
    $stmt = $db->prepare("SELECT MIN(proxima_cita) as proxima 
                          FROM visitas 
                          WHERE mascota_id = :id AND proxima_cita >= CURDATE()");
    $stmt->execute(['id' => $mascota_id]);
    $proxima_cita = $stmt->fetchColumn();
    
    // Calcular edad
    $fecha_nac = new DateTime($mascota['fecha_nacimiento']);
    $hoy = new DateTime();
    $edad = $hoy->diff($fecha_nac);
    $edad_texto = '';
    if ($edad->y > 0) {
        $edad_texto = $edad->y . ' año' . ($edad->y > 1 ? 's' : '');
    } elseif ($edad->m > 0) {
        $edad_texto = $edad->m . ' mes' . ($edad->m > 1 ? 'es' : '');
    } else {
        $edad_texto = $edad->d . ' día' . ($edad->d > 1 ? 's' : '');
    }
    
} catch (PDOException $e) {
    die("Error en la base de datos: " . $e->getMessage());
}

include '../includes/header.php';
?>

<div class="row mb-4">
    <div class="col-12">
        <nav aria-label="breadcrumb">
            <ol class="breadcrumb">
                <li class="breadcrumb-item"><a href="index.php">Inicio</a></li>
                <li class="breadcrumb-item"><a href="mascotas.php">Mascotas</a></li>
                <li class="breadcrumb-item"><a href="ver_mascota.php?id=<?php echo $mascota_id; ?>"><?php echo htmlspecialchars($mascota['nombre']); ?></a></li>
                <li class="breadcrumb-item active">Historial Clínico</li>
            </ol>
        </nav>
        <h1><i class="bi bi-clipboard2-pulse"></i> Historial Clínico</h1>
    </div>
</div>

<!-- Información de la Mascota -->
<div class="row mb-4">
    <div class="col-md-12">
        <div class="card border-primary">
            <div class="card-header bg-primary text-white">
                <h5 class="mb-0"><i class="bi bi-heart-fill"></i> Información del Paciente</h5>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-2 text-center">
                        <?php if ($mascota['foto']): ?>
                            <img src="<?php echo BASE_URL; ?>/uploads/mascotas/<?php echo htmlspecialchars($mascota['foto']); ?>" 
                                 class="img-fluid rounded-circle" alt="Foto" style="max-width: 120px;">
                        <?php else: ?>
                            <div class="bg-secondary rounded-circle d-flex align-items-center justify-content-center" 
                                 style="width: 120px; height: 120px; margin: 0 auto;">
                                <i class="bi bi-image text-white" style="font-size: 3rem;"></i>
                            </div>
                        <?php endif; ?>
                    </div>
                    <div class="col-md-5">
                        <h4><?php echo htmlspecialchars($mascota['nombre']); ?></h4>
                        <p class="mb-1"><strong>Tipo:</strong> <?php echo htmlspecialchars($mascota['tipo_mascota']); ?></p>
                        <p class="mb-1"><strong>Raza:</strong> <?php echo htmlspecialchars($mascota['raza']); ?></p>
                        <p class="mb-1"><strong>Edad:</strong> <?php echo $edad_texto; ?></p>
                        <p class="mb-1"><strong>Peso Actual:</strong> <?php echo $mascota['peso'] ? number_format($mascota['peso'], 1) . ' kg' : 'No registrado'; ?></p>
                        <p class="mb-1"><strong>Código:</strong> <code><?php echo htmlspecialchars($mascota['codigo_barras']); ?></code></p>
                    </div>
                    <div class="col-md-5">
                        <h5>Dueño</h5>
                        <p class="mb-1"><strong>Nombre:</strong> <?php echo htmlspecialchars($mascota['dueno_nombre']); ?></p>
                        <p class="mb-1"><strong>Cédula:</strong> <?php echo htmlspecialchars($mascota['cedula']); ?></p>
                        <p class="mb-1"><strong>Teléfono:</strong> <?php echo htmlspecialchars($mascota['telefono']); ?></p>
                        <?php if ($mascota['email']): ?>
                            <p class="mb-1"><strong>Email:</strong> <?php echo htmlspecialchars($mascota['email']); ?></p>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Estadísticas -->
<div class="row mb-4">
    <div class="col-md-3">
        <div class="card bg-primary text-white">
            <div class="card-body text-center">
                <h3 class="mb-0"><?php echo $total_visitas; ?></h3>
                <p class="mb-0">Total Visitas</p>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card bg-success text-white">
            <div class="card-body text-center">
                <h3 class="mb-0"><?php echo $total_vacunas; ?></h3>
                <p class="mb-0">Vacunas Aplicadas</p>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card bg-info text-white">
            <div class="card-body text-center">
                <h3 class="mb-0"><?php echo $ultima_visita ? date('d/m/Y', strtotime($ultima_visita['fecha_visita'])) : 'N/A'; ?></h3>
                <p class="mb-0">Última Visita</p>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card bg-warning text-dark">
            <div class="card-body text-center">
                <h3 class="mb-0"><?php echo $proxima_cita ? date('d/m/Y', strtotime($proxima_cita)) : 'Sin cita'; ?></h3>
                <p class="mb-0">Próxima Cita</p>
            </div>
        </div>
    </div>
</div>

<!-- Tabs de Historial -->
<ul class="nav nav-tabs mb-3" role="tablist">
    <li class="nav-item" role="presentation">
        <button class="nav-link active" id="visitas-tab" data-bs-toggle="tab" data-bs-target="#visitas" type="button">
            <i class="bi bi-clipboard-check"></i> Visitas (<?php echo $total_visitas; ?>)
        </button>
    </li>
    <li class="nav-item" role="presentation">
        <button class="nav-link" id="vacunas-tab" data-bs-toggle="tab" data-bs-target="#vacunas" type="button">
            <i class="bi bi-shield-check"></i> Vacunas (<?php echo $total_vacunas; ?>)
        </button>
    </li>
    <li class="nav-item" role="presentation">
        <button class="nav-link" id="peso-tab" data-bs-toggle="tab" data-bs-target="#peso" type="button">
            <i class="bi bi-graph-up"></i> Evolución del Peso
        </button>
    </li>
</ul>

<div class="tab-content">
    <!-- Tab de Visitas -->
    <div class="tab-pane fade show active" id="visitas" role="tabpanel">
        <?php if (empty($visitas)): ?>
            <div class="alert alert-info">
                <i class="bi bi-info-circle"></i> No hay visitas registradas para esta mascota.
            </div>
        <?php else: ?>
            <div class="timeline">
                <?php foreach ($visitas as $visita): ?>
                    <div class="card mb-3">
                        <div class="card-header bg-light">
                            <div class="row">
                                <div class="col-md-6">
                                    <h5 class="mb-0">
                                        <i class="bi bi-calendar-event"></i> 
                                        <?php echo date('d/m/Y H:i', strtotime($visita['fecha_visita'])); ?>
                                    </h5>
                                </div>
                                <div class="col-md-6 text-end">
                                    <span class="badge bg-primary">
                                        <i class="bi bi-person"></i> <?php echo htmlspecialchars($visita['profesional_nombre']); ?>
                                    </span>
                                    <?php if ($visita['peso_actual']): ?>
                                        <span class="badge bg-info">
                                            <i class="bi bi-speedometer"></i> <?php echo number_format($visita['peso_actual'], 1); ?> kg
                                        </span>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                        <div class="card-body">
                            <div class="row">
                                <div class="col-md-12">
                                    <p class="mb-2"><strong>Motivo:</strong> <?php echo nl2br(htmlspecialchars($visita['motivo'])); ?></p>
                                    <?php if ($visita['diagnostico']): ?>
                                        <p class="mb-2"><strong>Diagnóstico:</strong> <?php echo nl2br(htmlspecialchars($visita['diagnostico'])); ?></p>
                                    <?php endif; ?>
                                    <?php if ($visita['observaciones']): ?>
                                        <p class="mb-2"><strong>Observaciones:</strong> <?php echo nl2br(htmlspecialchars($visita['observaciones'])); ?></p>
                                    <?php endif; ?>
                                    <?php if ($visita['proxima_cita']): ?>
                                        <p class="mb-0"><strong>Próxima Cita:</strong> 
                                            <span class="badge bg-warning text-dark">
                                                <?php echo date('d/m/Y', strtotime($visita['proxima_cita'])); ?>
                                            </span>
                                        </p>
                                    <?php endif; ?>
                                </div>
                            </div>
                            <div class="mt-3">
                                <a href="detalle_visita.php?id=<?php echo $visita['id']; ?>" class="btn btn-sm btn-primary">
                                    <i class="bi bi-eye"></i> Ver Detalles Completos
                                </a>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>
    
    <!-- Tab de Vacunas -->
    <div class="tab-pane fade" id="vacunas" role="tabpanel">
        <?php if (empty($vacunas)): ?>
            <div class="alert alert-info">
                <i class="bi bi-info-circle"></i> No hay vacunas registradas para esta mascota.
            </div>
        <?php else: ?>
            <div class="table-responsive">
                <table class="table table-hover">
                    <thead class="table-success">
                        <tr>
                            <th>Fecha Aplicación</th>
                            <th>Vacuna</th>
                            <th>Próxima Dosis</th>
                            <th>Profesional</th>
                            <th>Estado</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($vacunas as $vacuna): ?>
                            <tr>
                                <td><?php echo date('d/m/Y', strtotime($vacuna['fecha_aplicacion'])); ?></td>
                                <td><strong><?php echo htmlspecialchars($vacuna['nombre_vacuna']); ?></strong></td>
                                <td>
                                    <?php if ($vacuna['proxima_dosis']): ?>
                                        <?php 
                                        $proxima = new DateTime($vacuna['proxima_dosis']);
                                        $hoy = new DateTime();
                                        $vencida = $proxima < $hoy;
                                        ?>
                                        <span class="badge <?php echo $vencida ? 'bg-danger' : 'bg-success'; ?>">
                                            <?php echo date('d/m/Y', strtotime($vacuna['proxima_dosis'])); ?>
                                        </span>
                                    <?php else: ?>
                                        <span class="text-muted">No programada</span>
                                    <?php endif; ?>
                                </td>
                                <td><?php echo htmlspecialchars($vacuna['profesional_nombre']); ?></td>
                                <td>
                                    <?php if ($vacuna['proxima_dosis']): ?>
                                        <?php if ($vencida): ?>
                                            <span class="badge bg-danger">Vencida</span>
                                        <?php else: ?>
                                            <span class="badge bg-success">Al día</span>
                                        <?php endif; ?>
                                    <?php else: ?>
                                        <span class="badge bg-secondary">Completa</span>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>
    
    <!-- Tab de Evolución del Peso -->
    <div class="tab-pane fade" id="peso" role="tabpanel">
        <?php if (empty($pesos)): ?>
            <div class="alert alert-info">
                <i class="bi bi-info-circle"></i> No hay registros de peso para esta mascota.
            </div>
        <?php else: ?>
            <div class="card">
                <div class="card-body">
                    <canvas id="pesoChart" width="400" height="100"></canvas>
                </div>
            </div>
            <div class="table-responsive mt-3">
                <table class="table table-sm">
                    <thead class="table-info">
                        <tr>
                            <th>Fecha</th>
                            <th>Peso (kg)</th>
                            <th>Variación</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php 
                        $peso_anterior = null;
                        foreach ($pesos as $index => $peso): 
                            $variacion = null;
                            if ($peso_anterior !== null) {
                                $variacion = $peso['peso_actual'] - $peso_anterior;
                            }
                            $peso_anterior = $peso['peso_actual'];
                        ?>
                            <tr>
                                <td><?php echo date('d/m/Y', strtotime($peso['fecha_visita'])); ?></td>
                                <td><strong><?php echo number_format($peso['peso_actual'], 1); ?> kg</strong></td>
                                <td>
                                    <?php if ($variacion !== null): ?>
                                        <?php if ($variacion > 0): ?>
                                            <span class="text-success">
                                                <i class="bi bi-arrow-up"></i> +<?php echo number_format($variacion, 1); ?> kg
                                            </span>
                                        <?php elseif ($variacion < 0): ?>
                                            <span class="text-danger">
                                                <i class="bi bi-arrow-down"></i> <?php echo number_format($variacion, 1); ?> kg
                                            </span>
                                        <?php else: ?>
                                            <span class="text-muted">Sin cambio</span>
                                        <?php endif; ?>
                                    <?php else: ?>
                                        <span class="text-muted">-</span>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>
</div>

<div class="row mt-4">
    <div class="col-12">
        <a href="ver_mascota.php?id=<?php echo $mascota_id; ?>" class="btn btn-secondary">
            <i class="bi bi-arrow-left"></i> Volver
        </a>
        <a href="nueva_visita.php?mascota_id=<?php echo $mascota_id; ?>" class="btn btn-primary">
            <i class="bi bi-plus-circle"></i> Nueva Visita
        </a>
        <a href="generar_reporte_pdf.php?visita_id=ultima&mascota_id=<?php echo $mascota_id; ?>" class="btn btn-success" target="_blank">
            <i class="bi bi-file-pdf"></i> Exportar Historial PDF
        </a>
    </div>
</div>

<?php if (!empty($pesos)): ?>
<script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    const ctx = document.getElementById('pesoChart').getContext('2d');
    
    const fechas = <?php echo json_encode(array_map(function($p) { 
        return date('d/m/Y', strtotime($p['fecha_visita'])); 
    }, $pesos)); ?>;
    
    const pesos = <?php echo json_encode(array_map(function($p) { 
        return (float)$p['peso_actual']; 
    }, $pesos)); ?>;
    
    new Chart(ctx, {
        type: 'line',
        data: {
            labels: fechas,
            datasets: [{
                label: 'Peso (kg)',
                data: pesos,
                borderColor: 'rgb(75, 192, 192)',
                backgroundColor: 'rgba(75, 192, 192, 0.2)',
                tension: 0.1,
                fill: true
            }]
        },
        options: {
            responsive: true,
            plugins: {
                title: {
                    display: true,
                    text: 'Evolución del Peso'
                },
                legend: {
                    display: false
                }
            },
            scales: {
                y: {
                    beginAtZero: false,
                    title: {
                        display: true,
                        text: 'Peso (kg)'
                    }
                },
                x: {
                    title: {
                        display: true,
                        text: 'Fecha'
                    }
                }
            }
        }
    });
});
</script>
<?php endif; ?>

<?php include '../includes/footer.php'; ?>
