<?php
/**
 * Detalle de Visita Individual
 * Sistema de Administración de Veterinaria
 */

require_once '../config/config.php';
requerir_autenticacion();

if (!tiene_rol('colaborador') && !tiene_rol('administrador')) {
    header('Location: ' . BASE_URL . '/index.php');
    exit();
}

$titulo_pagina = 'Detalle de Visita';

$visita_id = (int)($_GET['id'] ?? 0);

if ($visita_id <= 0) {
    redirigir_con_mensaje('index.php', 'ID de visita inválido', 'error');
}

try {
    $db = Database::getInstance()->getConnection();
    
    // Obtener datos de la visita
    $stmt = $db->prepare("SELECT v.*, u.nombre as profesional_nombre, u.email as profesional_email,
                          m.nombre as mascota_nombre, m.tipo_mascota, m.raza, m.foto,
                          d.nombre_completo as dueno_nombre, d.telefono, d.cedula
                          FROM visitas v
                          INNER JOIN usuarios u ON v.profesional_id = u.id
                          INNER JOIN mascotas m ON v.mascota_id = m.id
                          INNER JOIN duenos d ON m.dueno_id = d.id
                          WHERE v.id = :id");
    $stmt->execute(['id' => $visita_id]);
    $visita = $stmt->fetch();
    
    if (!$visita) {
        redirigir_con_mensaje('index.php', 'Visita no encontrada', 'error');
    }
    
    $mascota_id = $visita['mascota_id'];
    
    // Obtener tratamientos de la visita
    $stmt = $db->prepare("SELECT vt.*, t.nombre as tratamiento_nombre, t.descripcion
                          FROM visita_tratamientos vt
                          INNER JOIN tratamientos t ON vt.tratamiento_id = t.id
                          WHERE vt.visita_id = :visita_id");
    $stmt->execute(['visita_id' => $visita_id]);
    $tratamientos = $stmt->fetchAll();
    
    // Obtener medicamentos de la visita
    $stmt = $db->prepare("SELECT vm.*, m.nombre as medicamento_nombre, m.descripcion
                          FROM visita_medicamentos vm
                          INNER JOIN medicamentos m ON vm.medicamento_id = m.id
                          WHERE vm.visita_id = :visita_id");
    $stmt->execute(['visita_id' => $visita_id]);
    $medicamentos = $stmt->fetchAll();
    
    // Obtener factura
    $stmt = $db->prepare("SELECT * FROM facturas WHERE visita_id = :visita_id");
    $stmt->execute(['visita_id' => $visita_id]);
    $factura = $stmt->fetch();
    
    // Calcular totales
    $total_tratamientos = array_sum(array_column($tratamientos, 'subtotal'));
    $total_medicamentos = array_sum(array_column($medicamentos, 'subtotal'));
    $total_general = $total_tratamientos + $total_medicamentos;
    
} catch (PDOException $e) {
    die("Error en la base de datos: " . $e->getMessage());
}

include '../includes/header.php';
?>

<div class="row mb-4">
    <div class="col-12">
        <nav aria-label="breadcrumb">
            <ol class="breadcrumb">
                <li class="breadcrumb-item"><a href="index.php">Inicio</a></li>
                <li class="breadcrumb-item"><a href="mascotas.php">Mascotas</a></li>
                <li class="breadcrumb-item"><a href="ver_mascota.php?id=<?php echo $mascota_id; ?>"><?php echo htmlspecialchars($visita['mascota_nombre']); ?></a></li>
                <li class="breadcrumb-item"><a href="historial_clinico.php?id=<?php echo $mascota_id; ?>">Historial Clínico</a></li>
                <li class="breadcrumb-item active">Detalle de Visita</li>
            </ol>
        </nav>
        <h1><i class="bi bi-clipboard-check"></i> Detalle de Visita</h1>
    </div>
</div>

<!-- Información de la Visita -->
<div class="row mb-4">
    <div class="col-md-8">
        <div class="card">
            <div class="card-header bg-primary text-white">
                <h5 class="mb-0"><i class="bi bi-calendar-event"></i> Información de la Visita</h5>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6">
                        <p class="mb-2"><strong>Fecha y Hora:</strong><br>
                            <?php echo date('d/m/Y H:i', strtotime($visita['fecha_visita'])); ?>
                        </p>
                        <p class="mb-2"><strong>Profesional:</strong><br>
                            <?php echo htmlspecialchars($visita['profesional_nombre']); ?>
                        </p>
                        <?php if ($visita['peso_actual']): ?>
                            <p class="mb-2"><strong>Peso Registrado:</strong><br>
                                <span class="badge bg-info" style="font-size: 1rem;">
                                    <?php echo number_format($visita['peso_actual'], 1); ?> kg
                                </span>
                            </p>
                        <?php endif; ?>
                    </div>
                    <div class="col-md-6">
                        <p class="mb-2"><strong>Mascota:</strong><br>
                            <?php echo htmlspecialchars($visita['mascota_nombre']); ?> 
                            (<?php echo htmlspecialchars($visita['tipo_mascota']); ?>)
                        </p>
                        <p class="mb-2"><strong>Dueño:</strong><br>
                            <?php echo htmlspecialchars($visita['dueno_nombre']); ?>
                        </p>
                        <?php if ($visita['proxima_cita']): ?>
                            <p class="mb-2"><strong>Próxima Cita:</strong><br>
                                <span class="badge bg-warning text-dark" style="font-size: 1rem;">
                                    <?php echo date('d/m/Y', strtotime($visita['proxima_cita'])); ?>
                                </span>
                            </p>
                        <?php endif; ?>
                    </div>
                </div>
                
                <hr>
                
                <div class="mb-3">
                    <h6><strong>Motivo de la Visita:</strong></h6>
                    <p class="ms-3"><?php echo nl2br(htmlspecialchars($visita['motivo'])); ?></p>
                </div>
                
                <?php if ($visita['diagnostico']): ?>
                    <div class="mb-3">
                        <h6><strong>Diagnóstico:</strong></h6>
                        <p class="ms-3"><?php echo nl2br(htmlspecialchars($visita['diagnostico'])); ?></p>
                    </div>
                <?php endif; ?>
                
                <?php if ($visita['observaciones']): ?>
                    <div class="mb-3">
                        <h6><strong>Observaciones:</strong></h6>
                        <p class="ms-3"><?php echo nl2br(htmlspecialchars($visita['observaciones'])); ?></p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <div class="col-md-4">
        <?php if ($visita['foto']): ?>
            <div class="card mb-3">
                <div class="card-header bg-secondary text-white">
                    <h6 class="mb-0"><i class="bi bi-image"></i> Foto de la Mascota</h6>
                </div>
                <div class="card-body text-center">
                    <img src="<?php echo BASE_URL; ?>/uploads/mascotas/<?php echo htmlspecialchars($visita['foto']); ?>" 
                         class="img-fluid rounded" alt="Foto">
                </div>
            </div>
        <?php endif; ?>
        
        <?php if ($factura): ?>
            <div class="card">
                <div class="card-header bg-success text-white">
                    <h6 class="mb-0"><i class="bi bi-receipt"></i> Factura</h6>
                </div>
                <div class="card-body">
                    <p class="mb-1"><strong>Número:</strong> <?php echo htmlspecialchars($factura['numero_factura']); ?></p>
                    <p class="mb-1"><strong>Fecha:</strong> <?php echo date('d/m/Y', strtotime($factura['fecha_factura'])); ?></p>
                    <p class="mb-1"><strong>Estado:</strong> 
                        <?php
                        $estado_class = [
                            'pendiente' => 'warning',
                            'pagada' => 'success',
                            'cancelada' => 'danger'
                        ];
                        $estado_texto = [
                            'pendiente' => 'Pendiente',
                            'pagada' => 'Pagada',
                            'cancelada' => 'Cancelada'
                        ];
                        ?>
                        <span class="badge bg-<?php echo $estado_class[$factura['estado']]; ?>">
                            <?php echo $estado_texto[$factura['estado']]; ?>
                        </span>
                    </p>
                    <hr>
                    <p class="mb-1">Subtotal: <?php echo formatear_moneda($factura['subtotal']); ?></p>
                    <?php if ($factura['descuento'] > 0): ?>
                        <p class="mb-1">Descuento: -<?php echo formatear_moneda($factura['descuento']); ?></p>
                    <?php endif; ?>
                    <h5 class="mb-0">Total: <?php echo formatear_moneda($factura['total']); ?></h5>
                </div>
            </div>
        <?php endif; ?>
    </div>
</div>

<!-- Tratamientos Aplicados -->
<?php if (!empty($tratamientos)): ?>
    <div class="row mb-4">
        <div class="col-12">
            <div class="card">
                <div class="card-header bg-info text-white">
                    <h5 class="mb-0"><i class="bi bi-clipboard-pulse"></i> Tratamientos Aplicados</h5>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead class="table-light">
                                <tr>
                                    <th>Tratamiento</th>
                                    <th>Descripción</th>
                                    <th class="text-center">Cantidad</th>
                                    <th class="text-end">Precio Unit.</th>
                                    <th class="text-end">Subtotal</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($tratamientos as $trat): ?>
                                    <tr>
                                        <td><strong><?php echo htmlspecialchars($trat['tratamiento_nombre']); ?></strong></td>
                                        <td><?php echo htmlspecialchars($trat['descripcion'] ?? '-'); ?></td>
                                        <td class="text-center"><?php echo $trat['cantidad']; ?></td>
                                        <td class="text-end"><?php echo formatear_moneda($trat['precio_unitario']); ?></td>
                                        <td class="text-end"><strong><?php echo formatear_moneda($trat['subtotal']); ?></strong></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                            <tfoot class="table-light">
                                <tr>
                                    <th colspan="4" class="text-end">Total Tratamientos:</th>
                                    <th class="text-end"><?php echo formatear_moneda($total_tratamientos); ?></th>
                                </tr>
                            </tfoot>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
<?php endif; ?>

<!-- Medicamentos Recetados -->
<?php if (!empty($medicamentos)): ?>
    <div class="row mb-4">
        <div class="col-12">
            <div class="card">
                <div class="card-header bg-warning text-dark">
                    <h5 class="mb-0"><i class="bi bi-capsule"></i> Medicamentos Recetados</h5>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead class="table-light">
                                <tr>
                                    <th>Medicamento</th>
                                    <th>Descripción</th>
                                    <th class="text-center">Cantidad</th>
                                    <th class="text-end">Precio Unit.</th>
                                    <th class="text-end">Subtotal</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($medicamentos as $med): ?>
                                    <tr>
                                        <td><strong><?php echo htmlspecialchars($med['medicamento_nombre']); ?></strong></td>
                                        <td><?php echo htmlspecialchars($med['descripcion'] ?? '-'); ?></td>
                                        <td class="text-center"><?php echo $med['cantidad']; ?></td>
                                        <td class="text-end"><?php echo formatear_moneda($med['precio_unitario']); ?></td>
                                        <td class="text-end"><strong><?php echo formatear_moneda($med['subtotal']); ?></strong></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                            <tfoot class="table-light">
                                <tr>
                                    <th colspan="4" class="text-end">Total Medicamentos:</th>
                                    <th class="text-end"><?php echo formatear_moneda($total_medicamentos); ?></th>
                                </tr>
                            </tfoot>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
<?php endif; ?>

<!-- Resumen Total -->
<?php if (!empty($tratamientos) || !empty($medicamentos)): ?>
    <div class="row mb-4">
        <div class="col-md-6 offset-md-6">
            <div class="card border-success">
                <div class="card-body">
                    <h5 class="card-title">Resumen de Costos</h5>
                    <table class="table table-sm mb-0">
                        <tr>
                            <td>Tratamientos:</td>
                            <td class="text-end"><?php echo formatear_moneda($total_tratamientos); ?></td>
                        </tr>
                        <tr>
                            <td>Medicamentos:</td>
                            <td class="text-end"><?php echo formatear_moneda($total_medicamentos); ?></td>
                        </tr>
                        <?php if ($factura && $factura['descuento'] > 0): ?>
                            <tr>
                                <td>Descuento:</td>
                                <td class="text-end text-danger">-<?php echo formatear_moneda($factura['descuento']); ?></td>
                            </tr>
                        <?php endif; ?>
                        <tr class="table-success">
                            <th>Total General:</th>
                            <th class="text-end">
                                <?php echo formatear_moneda($factura ? $factura['total'] : $total_general); ?>
                            </th>
                        </tr>
                    </table>
                </div>
            </div>
        </div>
    </div>
<?php endif; ?>

<!-- Botones de Acción -->
<div class="row">
    <div class="col-12">
        <a href="historial_clinico.php?id=<?php echo $mascota_id; ?>" class="btn btn-secondary">
            <i class="bi bi-arrow-left"></i> Volver al Historial
        </a>
        <a href="ver_mascota.php?id=<?php echo $mascota_id; ?>" class="btn btn-primary">
            <i class="bi bi-heart"></i> Ver Ficha de Mascota
        </a>
        <a href="generar_reporte_pdf.php?visita_id=<?php echo $visita_id; ?>" class="btn btn-success" target="_blank">
            <i class="bi bi-file-pdf"></i> Exportar PDF
        </a>
    </div>
</div>

<?php include '../includes/footer.php'; ?>
