<?php
/**
 * Búsqueda de Mascotas y Dueños
 * Sistema de Administración de Veterinaria
 */

require_once '../config/config.php';
requerir_autenticacion();

if (!tiene_rol('colaborador') && !tiene_rol('administrador')) {
    header('Location: ' . BASE_URL . '/index.php');
    exit();
}

$titulo_pagina = 'Buscar Mascotas';

$resultados = [];
$termino_busqueda = '';
$mensaje_debug = '';

if (isset($_GET['buscar']) && !empty($_GET['termino'])) {
    $termino_busqueda = trim($_GET['termino']);
    
    try {
        $db = Database::getInstance()->getConnection();
        
        // Buscar por nombre de mascota, nombre de dueño, teléfono, cédula o código de barras
        $sql = "SELECT m.*, 
                       d.nombre_completo as dueno_nombre, 
                       d.telefono, 
                       d.cedula, 
                       d.tiene_whatsapp, 
                       d.tiene_telegram, 
                       d.direccion,
                       d.email
                FROM mascotas m
                INNER JOIN duenos d ON m.dueno_id = d.id
                WHERE m.activo = 1 AND (
                    m.nombre LIKE :termino1 
                    OR d.nombre_completo LIKE :termino2
                    OR d.telefono LIKE :termino3
                    OR d.cedula LIKE :termino4
                    OR m.codigo_barras LIKE :termino5
                )
                ORDER BY m.nombre";
        
        $stmt = $db->prepare($sql);
        $termino_like = '%' . $termino_busqueda . '%';
        $stmt->execute([
            'termino1' => $termino_like,
            'termino2' => $termino_like,
            'termino3' => $termino_like,
            'termino4' => $termino_like,
            'termino5' => $termino_like
        ]);
        $resultados = $stmt->fetchAll();
        
        $mensaje_debug = "Búsqueda: '$termino_busqueda' - Resultados encontrados: " . count($resultados);
        
    } catch (PDOException $e) {
        $error = "Error en la búsqueda: " . $e->getMessage();
        $mensaje_debug = $error;
    }
}

include '../includes/header.php';
?>

<div class="row mb-4">
    <div class="col-12">
        <h1><i class="bi bi-search"></i> Buscar Mascotas</h1>
        <p class="text-muted">Busque por nombre de mascota, nombre del dueño, teléfono, cédula o código de barras</p>
    </div>
</div>

<!-- Formulario de búsqueda -->
<div class="card mb-4">
    <div class="card-body">
        <form method="GET" action="">
            <div class="input-group input-group-lg">
                <span class="input-group-text"><i class="bi bi-search"></i></span>
                <input type="text" class="form-control" name="termino" 
                       placeholder="Ingrese nombre, teléfono, cédula o código..." 
                       value="<?php echo htmlspecialchars($termino_busqueda); ?>" 
                       autofocus required>
                <button class="btn btn-primary" type="submit" name="buscar" value="1">
                    <i class="bi bi-search"></i> Buscar
                </button>
            </div>
        </form>
    </div>
</div>

<?php if ($mensaje_debug && tiene_rol('administrador')): ?>
    <div class="alert alert-info">
        <small><strong>Debug:</strong> <?php echo htmlspecialchars($mensaje_debug); ?></small>
    </div>
<?php endif; ?>

<?php if (isset($_GET['buscar'])): ?>
    <?php if (count($resultados) > 0): ?>
        <div class="alert alert-success">
            <i class="bi bi-check-circle"></i> Se encontraron <strong><?php echo count($resultados); ?></strong> resultado(s)
        </div>
        
        <div class="row">
            <?php foreach ($resultados as $mascota): ?>
                <?php
                $edad = '';
                if ($mascota['fecha_nacimiento']) {
                    $fecha_nac = new DateTime($mascota['fecha_nacimiento']);
                    $hoy = new DateTime();
                    $diff = $hoy->diff($fecha_nac);
                    if ($diff->y > 0) {
                        $edad = $diff->y . ' año' . ($diff->y > 1 ? 's' : '');
                    } else {
                        $edad = $diff->m . ' mes' . ($diff->m > 1 ? 'es' : '');
                    }
                }
                ?>
                <div class="col-md-6 mb-3">
                    <div class="card h-100">
                        <div class="card-body">
                            <div class="row">
                                <div class="col-md-3 text-center">
                                    <?php if ($mascota['foto']): ?>
                                        <img src="<?php echo BASE_URL; ?>/uploads/mascotas/<?php echo htmlspecialchars($mascota['foto']); ?>" 
                                             class="img-fluid rounded" 
                                             alt="Foto de <?php echo htmlspecialchars($mascota['nombre']); ?>"
                                             style="max-height: 100px; object-fit: cover;">
                                    <?php else: ?>
                                        <i class="bi bi-heart-fill text-muted" style="font-size: 4rem;"></i>
                                    <?php endif; ?>
                                </div>
                                <div class="col-md-9">
                                    <h5 class="card-title">
                                        <i class="bi bi-heart text-danger"></i> 
                                        <?php echo htmlspecialchars($mascota['nombre']); ?>
                                    </h5>
                                    <p class="mb-1">
                                        <strong>Tipo:</strong> <?php echo ucfirst($mascota['tipo_mascota']); ?>
                                        <?php if ($mascota['raza']): ?>
                                            | <strong>Raza:</strong> <?php echo htmlspecialchars($mascota['raza']); ?>
                                        <?php endif; ?>
                                    </p>
                                    <?php if ($edad): ?>
                                        <p class="mb-1"><strong>Edad:</strong> <?php echo $edad; ?></p>
                                    <?php endif; ?>
                                    <p class="mb-1"><strong>Código:</strong> <code><?php echo htmlspecialchars($mascota['codigo_barras']); ?></code></p>
                                    
                                    <hr>
                                    
                                    <h6><i class="bi bi-person"></i> Dueño</h6>
                                    <p class="mb-1"><strong><?php echo htmlspecialchars($mascota['dueno_nombre']); ?></strong></p>
                                    <p class="mb-1">
                                        <i class="bi bi-telephone"></i> 
                                        <a href="<?php echo crear_enlace_whatsapp($mascota['telefono']); ?>" target="_blank" class="text-decoration-none">
                                            <?php echo htmlspecialchars($mascota['telefono']); ?>
                                        </a>
                                        <?php if ($mascota['tiene_whatsapp']): ?>
                                            <a href="<?php echo crear_enlace_whatsapp($mascota['telefono']); ?>" 
                                               target="_blank" class="btn btn-sm btn-success ms-2">
                                                <i class="bi bi-whatsapp"></i> WhatsApp
                                            </a>
                                        <?php endif; ?>
                                        <?php if ($mascota['tiene_telegram']): ?>
                                            <a href="<?php echo crear_enlace_telegram($mascota['telefono']); ?>" 
                                               target="_blank" class="btn btn-sm btn-info ms-2">
                                                <i class="bi bi-telegram"></i> Telegram
                                            </a>
                                        <?php endif; ?>
                                    </p>
                                    <?php if ($mascota['cedula']): ?>
                                        <p class="mb-1"><i class="bi bi-card-text"></i> CI: <?php echo htmlspecialchars($mascota['cedula']); ?></p>
                                    <?php endif; ?>
                                    <?php if ($mascota['direccion']): ?>
                                        <p class="mb-1"><i class="bi bi-geo-alt"></i> <?php echo htmlspecialchars($mascota['direccion']); ?></p>
                                    <?php endif; ?>
                                    
                                    <div class="mt-3">
                                        <a href="ver_mascota.php?id=<?php echo $mascota['id']; ?>" class="btn btn-primary btn-sm">
                                            <i class="bi bi-eye"></i> Ver Detalles
                                        </a>
                                        <a href="nueva_visita.php?mascota_id=<?php echo $mascota['id']; ?>" class="btn btn-success btn-sm">
                                            <i class="bi bi-plus-circle"></i> Nueva Visita
                                        </a>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
    <?php else: ?>
        <div class="alert alert-warning">
            <i class="bi bi-exclamation-triangle"></i> No se encontraron resultados para "<strong><?php echo htmlspecialchars($termino_busqueda); ?></strong>"
        </div>
        
        <div class="card">
            <div class="card-body text-center py-5">
                <i class="bi bi-search" style="font-size: 4rem; color: #ccc;"></i>
                <h5 class="mt-3">¿No encuentra la mascota?</h5>
                <p class="text-muted">Puede registrar una nueva mascota</p>
                <a href="nueva_mascota.php" class="btn btn-primary">
                    <i class="bi bi-plus-circle"></i> Registrar Nueva Mascota
                </a>
            </div>
        </div>
    <?php endif; ?>
<?php else: ?>
    <div class="card">
        <div class="card-body text-center py-5">
            <i class="bi bi-search" style="font-size: 5rem; color: #ccc;"></i>
            <h4 class="mt-3">Ingrese un término de búsqueda</h4>
            <p class="text-muted">Puede buscar por nombre de mascota, nombre del dueño, teléfono, cédula o código de barras</p>
            
            <div class="mt-4">
                <h6>Ejemplos de búsqueda:</h6>
                <ul class="list-unstyled">
                    <li><i class="bi bi-search"></i> <code>Max</code> - Busca por nombre de mascota</li>
                    <li><i class="bi bi-search"></i> <code>Juan Pérez</code> - Busca por nombre del dueño</li>
                    <li><i class="bi bi-search"></i> <code>0981234567</code> - Busca por teléfono</li>
                    <li><i class="bi bi-search"></i> <code>1234567</code> - Busca por cédula</li>
                </ul>
            </div>
        </div>
    </div>
<?php endif; ?>

<?php include '../includes/footer.php'; ?>
