/**
 * JavaScript Principal
 * Sistema de Administración de Veterinaria
 */

// Función para formatear números con puntos de miles
function formatearNumero(numero) {
    return new Intl.NumberFormat('es-PY').format(numero);
}

// Función para formatear moneda
function formatearMoneda(monto) {
    return 'Gs. ' + formatearNumero(monto);
}

// Auto-cerrar alertas después de 5 segundos
document.addEventListener('DOMContentLoaded', function() {
    const alerts = document.querySelectorAll('.alert:not(.alert-permanent)');
    alerts.forEach(function(alert) {
        setTimeout(function() {
            const bsAlert = new bootstrap.Alert(alert);
            bsAlert.close();
        }, 5000);
    });
});

// Confirmación antes de eliminar
function confirmarEliminacion(mensaje) {
    return confirm(mensaje || '¿Está seguro de que desea eliminar este elemento?');
}

// Validación de formularios
(function() {
    'use strict';
    const forms = document.querySelectorAll('.needs-validation');
    Array.from(forms).forEach(function(form) {
        form.addEventListener('submit', function(event) {
            if (!form.checkValidity()) {
                event.preventDefault();
                event.stopPropagation();
            }
            form.classList.add('was-validated');
        }, false);
    });
})();

// Función de búsqueda con autocompletado
function inicializarBuscador() {
    const inputBuscar = document.getElementById('buscarGlobal');
    if (!inputBuscar) return;
    
    let timeoutId;
    
    inputBuscar.addEventListener('input', function() {
        clearTimeout(timeoutId);
        const termino = this.value.trim();
        
        if (termino.length < 2) {
            ocultarResultados();
            return;
        }
        
        timeoutId = setTimeout(() => {
            buscarMascotas(termino);
        }, 300);
    });
}

function buscarMascotas(termino) {
    const baseUrl = document.querySelector('meta[name="base-url"]')?.content || '';
    
    fetch(`${baseUrl}/colaborador/api_buscar.php?q=${encodeURIComponent(termino)}`)
        .then(response => response.json())
        .then(data => {
            mostrarResultados(data);
        })
        .catch(error => {
            console.error('Error en la búsqueda:', error);
        });
}

function mostrarResultados(mascotas) {
    let contenedor = document.getElementById('resultadosBusqueda');
    if (!contenedor) {
        contenedor = document.createElement('div');
        contenedor.id = 'resultadosBusqueda';
        contenedor.className = 'list-group position-absolute w-100';
        contenedor.style.zIndex = '1000';
        contenedor.style.maxHeight = '400px';
        contenedor.style.overflowY = 'auto';
        document.getElementById('buscarGlobal').parentElement.appendChild(contenedor);
    }
    
    if (mascotas.length === 0) {
        contenedor.innerHTML = '<div class="list-group-item">No se encontraron resultados</div>';
        return;
    }
    
    const baseUrl = document.querySelector('meta[name="base-url"]')?.content || '';
    let html = '';
    
    mascotas.forEach(mascota => {
        html += `
            <a href="${baseUrl}/colaborador/ver_mascota.php?id=${mascota.id}" class="list-group-item list-group-item-action">
                <div class="d-flex align-items-center">
                    ${mascota.foto ? `<img src="${mascota.foto}" class="rounded me-2" style="width: 40px; height: 40px; object-fit: cover;">` : '<i class="bi bi-heart-fill me-2"></i>'}
                    <div>
                        <strong>${mascota.nombre}</strong> <small class="text-muted">(${mascota.tipo})</small><br>
                        <small>Dueño: ${mascota.dueno} - ${mascota.telefono}</small>
                    </div>
                </div>
            </a>
        `;
    });
    
    contenedor.innerHTML = html;
}

function ocultarResultados() {
    const contenedor = document.getElementById('resultadosBusqueda');
    if (contenedor) {
        contenedor.innerHTML = '';
    }
}

// Cerrar resultados al hacer clic fuera
document.addEventListener('click', function(e) {
    if (!e.target.closest('#buscarGlobal') && !e.target.closest('#resultadosBusqueda')) {
        ocultarResultados();
    }
});

// Inicializar búsqueda al cargar
document.addEventListener('DOMContentLoaded', function() {
    inicializarBuscador();
});

// ============================================
// SISTEMA DE ALERTAS DE CUMPLEAÑOS
// ============================================

/**
 * Verificar y mostrar cumpleaños del día
 */
function verificarCumpleanos() {
    // Solo para colaboradores y administradores
    const baseUrl = document.querySelector('meta[name="base-url"]')?.content || '';
    
    fetch(baseUrl + '/includes/api_cumpleanos.php')
        .then(response => response.json())
        .then(data => {
            if (data.success && data.total > 0) {
                mostrarModalCumpleanos(data.cumpleanos);
            }
        })
        .catch(error => {
            console.error('Error al verificar cumpleaños:', error);
        });
}

/**
 * Mostrar modal con felicitaciones de cumpleaños
 */
function mostrarModalCumpleanos(cumpleanos) {
    const baseUrl = document.querySelector('meta[name="base-url"]')?.content || '';
    let html = '';
    
    cumpleanos.forEach((mascota, index) => {
        const fotoUrl = mascota.foto 
            ? `${baseUrl}/uploads/mascotas/${mascota.foto}` 
            : `${baseUrl}/assets/images/default-pet.png`;
        
        const edad = mascota.edad;
        const edadTexto = edad === 1 ? '1 año' : `${edad} años`;
        
        html += `
            <div class="cumpleanos-card" style="animation-delay: ${index * 0.2}s">
                <div class="row align-items-center">
                    <div class="col-md-3 text-center">
                        <img src="${fotoUrl}" alt="${mascota.nombre}" class="cumpleanos-foto" onerror="this.src='${baseUrl}/assets/images/logo.png'">
                    </div>
                    <div class="col-md-6 text-center">
                        <div class="cumpleanos-emoji">🎉🎂🎈</div>
                        <div class="cumpleanos-nombre">${mascota.nombre}</div>
                        <p class="mb-1"><span class="badge bg-secondary">${mascota.tipo}</span></p>
                        <div class="cumpleanos-edad">${edadTexto}</div>
                        <p class="text-muted mb-0">Dueño: ${mascota.dueno_nombre}</p>
                    </div>
                    <div class="col-md-3 text-center">
                        <p class="mb-2"><strong>Enviar felicitación:</strong></p>
        `;
        
        if (mascota.tiene_whatsapp && mascota.telefono) {
            const telefono = mascota.telefono.replace(/\D/g, '');
            const mensaje = encodeURIComponent(`¡Feliz cumpleaños a ${mascota.nombre}! 🎉🎂 Que cumpla muchos años más lleno de salud y alegría. 🐾❤️`);
            html += `
                <a href="https://wa.me/${telefono}?text=${mensaje}" 
                   target="_blank" 
                   class="btn btn-whatsapp btn-sm mb-2 w-100">
                    <i class="bi bi-whatsapp"></i> WhatsApp
                </a>
            `;
        }
        
        if (mascota.tiene_telegram && mascota.telefono) {
            const telefono = mascota.telefono.replace(/\D/g, '');
            html += `
                <a href="https://t.me/${telefono}" 
                   target="_blank" 
                   class="btn btn-telegram btn-sm w-100">
                    <i class="bi bi-telegram"></i> Telegram
                </a>
            `;
        }
        
        if (!mascota.tiene_whatsapp && !mascota.tiene_telegram) {
            html += `<p class="text-muted small">Sin contacto disponible</p>`;
        }
        
        html += `
                    </div>
                </div>
            </div>
        `;
    });
    
    // Insertar contenido en el modal
    const modalContent = document.getElementById('cumpleanosContent');
    if (modalContent) {
        modalContent.innerHTML = html;
        
        // Mostrar el modal
        const modalElement = document.getElementById('modalCumpleanos');
        if (modalElement) {
            const modal = new bootstrap.Modal(modalElement);
            modal.show();
            
            // Reproducir sonido de celebración (opcional)
            reproducirSonidoCelebracion();
        }
    }
}

/**
 * Reproducir sonido de celebración (opcional)
 */
function reproducirSonidoCelebracion() {
    // Crear un beep simple usando Web Audio API
    try {
        const audioContext = new (window.AudioContext || window.webkitAudioContext)();
        const oscillator = audioContext.createOscillator();
        const gainNode = audioContext.createGain();
        
        oscillator.connect(gainNode);
        gainNode.connect(audioContext.destination);
        
        oscillator.frequency.value = 800;
        oscillator.type = 'sine';
        
        gainNode.gain.setValueAtTime(0.3, audioContext.currentTime);
        gainNode.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + 0.5);
        
        oscillator.start(audioContext.currentTime);
        oscillator.stop(audioContext.currentTime + 0.5);
    } catch (e) {
        // Silenciar errores de audio
    }
}

// Verificar cumpleaños al cargar la página (solo una vez por sesión)
document.addEventListener('DOMContentLoaded', function() {
    // Verificar si ya se mostró hoy
    const hoy = new Date().toDateString();
    const ultimaVerificacion = sessionStorage.getItem('cumpleanosVerificado');
    
    if (ultimaVerificacion !== hoy) {
        // Esperar 1 segundo para que la página cargue completamente
        setTimeout(function() {
            verificarCumpleanos();
            sessionStorage.setItem('cumpleanosVerificado', hoy);
        }, 1000);
    }
});
