<?php
/**
 * Gestión de Tratamientos y Tarifas
 * Sistema de Administración de Veterinaria
 */

require_once '../config/config.php';
requerir_rol('administrador');

$titulo_pagina = 'Gestión de Tratamientos y Tarifas';

// Procesar acciones
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $accion = $_POST['accion'] ?? '';
    
    try {
        $db = Database::getInstance()->getConnection();
        
        if ($accion === 'crear_tratamiento') {
            $nombre = limpiar_entrada($_POST['nombre']);
            $descripcion = limpiar_entrada($_POST['descripcion']);
            $tipo = limpiar_entrada($_POST['tipo']);
            $precio_base = (float)$_POST['precio_base'];
            
            $stmt = $db->prepare("INSERT INTO tratamientos (nombre, descripcion, tipo, precio_base) 
                                  VALUES (:nombre, :descripcion, :tipo, :precio_base)");
            $stmt->execute([
                'nombre' => $nombre,
                'descripcion' => $descripcion,
                'tipo' => $tipo,
                'precio_base' => $precio_base
            ]);
            
            redirigir_con_mensaje('tratamientos.php', 'Tratamiento creado exitosamente', 'success');
            
        } elseif ($accion === 'editar_tratamiento') {
            $id = (int)$_POST['id'];
            $nombre = limpiar_entrada($_POST['nombre']);
            $descripcion = limpiar_entrada($_POST['descripcion']);
            $tipo = limpiar_entrada($_POST['tipo']);
            $precio_base = (float)$_POST['precio_base'];
            $activo = isset($_POST['activo']) ? 1 : 0;
            
            $stmt = $db->prepare("UPDATE tratamientos 
                                  SET nombre = :nombre, descripcion = :descripcion, tipo = :tipo, 
                                      precio_base = :precio_base, activo = :activo
                                  WHERE id = :id");
            $stmt->execute([
                'nombre' => $nombre,
                'descripcion' => $descripcion,
                'tipo' => $tipo,
                'precio_base' => $precio_base,
                'activo' => $activo,
                'id' => $id
            ]);
            
            redirigir_con_mensaje('tratamientos.php', 'Tratamiento actualizado exitosamente', 'success');
            
        } elseif ($accion === 'eliminar_tratamiento') {
            $id = (int)$_POST['id'];
            $stmt = $db->prepare("DELETE FROM tratamientos WHERE id = :id");
            $stmt->execute(['id' => $id]);
            
            redirigir_con_mensaje('tratamientos.php', 'Tratamiento eliminado exitosamente', 'success');
            
        } elseif ($accion === 'guardar_tarifa') {
            $tratamiento_id = (int)$_POST['tratamiento_id'];
            $tipo_mascota = limpiar_entrada($_POST['tipo_mascota']);
            $precio = (float)$_POST['precio'];
            
            $stmt = $db->prepare("INSERT INTO tarifas (tratamiento_id, tipo_mascota, precio) 
                                  VALUES (:tratamiento_id, :tipo_mascota, :precio)
                                  ON DUPLICATE KEY UPDATE precio = :precio2");
            $stmt->execute([
                'tratamiento_id' => $tratamiento_id,
                'tipo_mascota' => $tipo_mascota,
                'precio' => $precio,
                'precio2' => $precio
            ]);
            
            redirigir_con_mensaje('tratamientos.php', 'Tarifa guardada exitosamente', 'success');
        }
        
    } catch (PDOException $e) {
        redirigir_con_mensaje('tratamientos.php', 'Error: ' . $e->getMessage(), 'danger');
    }
}

// Obtener tratamientos
try {
    $db = Database::getInstance()->getConnection();
    $stmt = $db->query("SELECT * FROM tratamientos ORDER BY nombre");
    $tratamientos = $stmt->fetchAll();
    
    // Obtener tarifas
    $stmt = $db->query("SELECT t.*, tr.nombre as tratamiento_nombre 
                        FROM tarifas t
                        INNER JOIN tratamientos tr ON t.tratamiento_id = tr.id
                        ORDER BY tr.nombre, t.tipo_mascota");
    $tarifas = $stmt->fetchAll();
    
    // Organizar tarifas por tratamiento
    $tarifas_por_tratamiento = [];
    foreach ($tarifas as $tarifa) {
        $tarifas_por_tratamiento[$tarifa['tratamiento_id']][$tarifa['tipo_mascota']] = $tarifa['precio'];
    }
    
} catch (PDOException $e) {
    die("Error en la base de datos: " . $e->getMessage());
}

$tipos_mascota = ['perro', 'gato', 'ave', 'conejo', 'hamster', 'reptil', 'otro'];
$tipos_tratamiento = ['baño', 'corte_pelo', 'corte_pluma', 'medicacion', 'analisis', 'consulta', 'cirugia', 'otro'];

include '../includes/header.php';
?>

<div class="row mb-4">
    <div class="col-md-6">
        <h1><i class="bi bi-clipboard-pulse"></i> Tratamientos y Tarifas</h1>
    </div>
    <div class="col-md-6 text-end">
        <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#modalNuevoTratamiento">
            <i class="bi bi-plus-circle"></i> Nuevo Tratamiento
        </button>
    </div>
</div>

<!-- Tabs -->
<ul class="nav nav-tabs mb-3" role="tablist">
    <li class="nav-item">
        <a class="nav-link active" data-bs-toggle="tab" href="#tabTratamientos">Tratamientos</a>
    </li>
    <li class="nav-item">
        <a class="nav-link" data-bs-toggle="tab" href="#tabTarifas">Tarifas por Tipo de Mascota</a>
    </li>
</ul>

<div class="tab-content">
    <!-- Tab Tratamientos -->
    <div class="tab-pane fade show active" id="tabTratamientos">
        <div class="card">
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-hover">
                        <thead>
                            <tr>
                                <th>Nombre</th>
                                <th>Tipo</th>
                                <th>Precio Base</th>
                                <th>Estado</th>
                                <th>Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($tratamientos as $trat): ?>
                                <tr>
                                    <td><strong><?php echo htmlspecialchars($trat['nombre']); ?></strong></td>
                                    <td><?php echo ucfirst(str_replace('_', ' ', $trat['tipo'])); ?></td>
                                    <td><?php echo formatear_moneda($trat['precio_base']); ?></td>
                                    <td>
                                        <?php if ($trat['activo']): ?>
                                            <span class="badge bg-success">Activo</span>
                                        <?php else: ?>
                                            <span class="badge bg-secondary">Inactivo</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <button type="button" class="btn btn-sm btn-warning" 
                                                onclick="editarTratamiento(<?php echo htmlspecialchars(json_encode($trat)); ?>)">
                                            <i class="bi bi-pencil"></i>
                                        </button>
                                        <button type="button" class="btn btn-sm btn-danger" 
                                                onclick="eliminarTratamiento(<?php echo $trat['id']; ?>, '<?php echo htmlspecialchars($trat['nombre']); ?>')">
                                            <i class="bi bi-trash"></i>
                                        </button>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Tab Tarifas -->
    <div class="tab-pane fade" id="tabTarifas">
        <div class="card">
            <div class="card-body">
                <p class="text-muted">Configure precios específicos para cada tipo de mascota. Si no se configura una tarifa específica, se usará el precio base del tratamiento.</p>
                
                <div class="table-responsive">
                    <table class="table table-bordered">
                        <thead>
                            <tr>
                                <th>Tratamiento</th>
                                <?php foreach ($tipos_mascota as $tipo): ?>
                                    <th><?php echo ucfirst($tipo); ?></th>
                                <?php endforeach; ?>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($tratamientos as $trat): ?>
                                <tr>
                                    <td><strong><?php echo htmlspecialchars($trat['nombre']); ?></strong></td>
                                    <?php foreach ($tipos_mascota as $tipo): ?>
                                        <td>
                                            <?php 
                                            $precio_actual = $tarifas_por_tratamiento[$trat['id']][$tipo] ?? $trat['precio_base'];
                                            $tiene_tarifa = isset($tarifas_por_tratamiento[$trat['id']][$tipo]);
                                            ?>
                                            <div class="input-group input-group-sm">
                                                <input type="number" class="form-control" 
                                                       value="<?php echo $precio_actual; ?>"
                                                       id="tarifa_<?php echo $trat['id']; ?>_<?php echo $tipo; ?>"
                                                       <?php echo $tiene_tarifa ? '' : 'style="background-color: #f8f9fa;"'; ?>>
                                                <button class="btn btn-outline-primary" type="button"
                                                        onclick="guardarTarifa(<?php echo $trat['id']; ?>, '<?php echo $tipo; ?>')">
                                                    <i class="bi bi-check"></i>
                                                </button>
                                            </div>
                                            <small class="text-muted">
                                                <?php echo $tiene_tarifa ? 'Personalizado' : 'Precio base'; ?>
                                            </small>
                                        </td>
                                    <?php endforeach; ?>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Modal Nuevo Tratamiento -->
<div class="modal fade" id="modalNuevoTratamiento" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST" action="">
                <input type="hidden" name="accion" value="crear_tratamiento">
                <div class="modal-header">
                    <h5 class="modal-title">Nuevo Tratamiento</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label">Nombre *</label>
                        <input type="text" class="form-control" name="nombre" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Descripción</label>
                        <textarea class="form-control" name="descripcion" rows="3"></textarea>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Tipo *</label>
                        <select class="form-select" name="tipo" required>
                            <?php foreach ($tipos_tratamiento as $tipo): ?>
                                <option value="<?php echo $tipo; ?>"><?php echo ucfirst(str_replace('_', ' ', $tipo)); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Precio Base (<?php echo MONEDA; ?>) *</label>
                        <input type="number" class="form-control" name="precio_base" step="1" required>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-primary">Guardar</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Modal Editar Tratamiento -->
<div class="modal fade" id="modalEditarTratamiento" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST" action="">
                <input type="hidden" name="accion" value="editar_tratamiento">
                <input type="hidden" name="id" id="edit_id">
                <div class="modal-header">
                    <h5 class="modal-title">Editar Tratamiento</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label">Nombre *</label>
                        <input type="text" class="form-control" name="nombre" id="edit_nombre" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Descripción</label>
                        <textarea class="form-control" name="descripcion" id="edit_descripcion" rows="3"></textarea>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Tipo *</label>
                        <select class="form-select" name="tipo" id="edit_tipo" required>
                            <?php foreach ($tipos_tratamiento as $tipo): ?>
                                <option value="<?php echo $tipo; ?>"><?php echo ucfirst(str_replace('_', ' ', $tipo)); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Precio Base (<?php echo MONEDA; ?>) *</label>
                        <input type="number" class="form-control" name="precio_base" id="edit_precio_base" step="1" required>
                    </div>
                    <div class="mb-3">
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" name="activo" id="edit_activo">
                            <label class="form-check-label" for="edit_activo">Activo</label>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-primary">Actualizar</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Formularios ocultos -->
<form method="POST" action="" id="formEliminar" style="display: none;">
    <input type="hidden" name="accion" value="eliminar_tratamiento">
    <input type="hidden" name="id" id="delete_id">
</form>

<form method="POST" action="" id="formGuardarTarifa" style="display: none;">
    <input type="hidden" name="accion" value="guardar_tarifa">
    <input type="hidden" name="tratamiento_id" id="tarifa_tratamiento_id">
    <input type="hidden" name="tipo_mascota" id="tarifa_tipo_mascota">
    <input type="hidden" name="precio" id="tarifa_precio">
</form>

<script>
function editarTratamiento(trat) {
    document.getElementById('edit_id').value = trat.id;
    document.getElementById('edit_nombre').value = trat.nombre;
    document.getElementById('edit_descripcion').value = trat.descripcion || '';
    document.getElementById('edit_tipo').value = trat.tipo;
    document.getElementById('edit_precio_base').value = trat.precio_base;
    document.getElementById('edit_activo').checked = trat.activo == 1;
    
    new bootstrap.Modal(document.getElementById('modalEditarTratamiento')).show();
}

function eliminarTratamiento(id, nombre) {
    if (confirm('¿Está seguro de eliminar el tratamiento "' + nombre + '"?')) {
        document.getElementById('delete_id').value = id;
        document.getElementById('formEliminar').submit();
    }
}

function guardarTarifa(tratamientoId, tipoMascota) {
    const inputId = 'tarifa_' + tratamientoId + '_' + tipoMascota;
    const precio = document.getElementById(inputId).value;
    
    document.getElementById('tarifa_tratamiento_id').value = tratamientoId;
    document.getElementById('tarifa_tipo_mascota').value = tipoMascota;
    document.getElementById('tarifa_precio').value = precio;
    document.getElementById('formGuardarTarifa').submit();
}
</script>

<?php include '../includes/footer.php'; ?>
