<?php
/**
 * Reportes y Estadísticas
 * Sistema de Administración de Veterinaria
 */

require_once '../config/config.php';
requerir_rol('administrador');

$titulo_pagina = 'Reportes y Estadísticas';

$periodo = $_GET['periodo'] ?? 'mes';
$fecha_desde = $_GET['fecha_desde'] ?? date('Y-m-01');
$fecha_hasta = $_GET['fecha_hasta'] ?? date('Y-m-d');

try {
    $db = Database::getInstance()->getConnection();
    
    // Ingresos por período
    $stmt = $db->prepare("SELECT 
                            DATE(fecha_factura) as fecha,
                            COUNT(*) as cantidad_facturas,
                            SUM(total) as total_ingresos
                         FROM facturas 
                         WHERE estado = 'pagada' 
                         AND fecha_factura BETWEEN :fecha_desde AND :fecha_hasta
                         GROUP BY DATE(fecha_factura)
                         ORDER BY fecha");
    $stmt->execute(['fecha_desde' => $fecha_desde, 'fecha_hasta' => $fecha_hasta]);
    $ingresos_diarios = $stmt->fetchAll();
    
    // Tratamientos más realizados
    $stmt = $db->prepare("SELECT 
                            t.nombre,
                            COUNT(vt.id) as cantidad,
                            SUM(vt.subtotal) as total_ingresos
                         FROM visita_tratamientos vt
                         INNER JOIN tratamientos t ON vt.tratamiento_id = t.id
                         INNER JOIN visitas v ON vt.visita_id = v.id
                         WHERE DATE(v.fecha_visita) BETWEEN :fecha_desde AND :fecha_hasta
                         GROUP BY t.id
                         ORDER BY cantidad DESC
                         LIMIT 10");
    $stmt->execute(['fecha_desde' => $fecha_desde, 'fecha_hasta' => $fecha_hasta]);
    $tratamientos_top = $stmt->fetchAll();
    
    // Medicamentos más utilizados
    $stmt = $db->prepare("SELECT 
                            m.nombre,
                            SUM(vm.cantidad) as cantidad_total,
                            SUM(vm.subtotal) as total_ingresos
                         FROM visita_medicamentos vm
                         INNER JOIN medicamentos m ON vm.medicamento_id = m.id
                         INNER JOIN visitas v ON vm.visita_id = v.id
                         WHERE DATE(v.fecha_visita) BETWEEN :fecha_desde AND :fecha_hasta
                         GROUP BY m.id
                         ORDER BY cantidad_total DESC
                         LIMIT 10");
    $stmt->execute(['fecha_desde' => $fecha_desde, 'fecha_hasta' => $fecha_hasta]);
    $medicamentos_top = $stmt->fetchAll();
    
    // Profesionales con más visitas
    $stmt = $db->prepare("SELECT 
                            u.nombre_completo,
                            COUNT(v.id) as cantidad_visitas,
                            COALESCE(SUM(f.total), 0) as total_ingresos
                         FROM usuarios u
                         LEFT JOIN visitas v ON u.id = v.profesional_id AND DATE(v.fecha_visita) BETWEEN :fecha_desde AND :fecha_hasta
                         LEFT JOIN facturas f ON v.id = f.visita_id AND f.estado = 'pagada'
                         WHERE u.rol = 'colaborador' AND u.activo = 1
                         GROUP BY u.id
                         ORDER BY cantidad_visitas DESC");
    $stmt->execute(['fecha_desde' => $fecha_desde, 'fecha_hasta' => $fecha_hasta]);
    $profesionales_stats = $stmt->fetchAll();
    
    // Resumen general
    $stmt = $db->prepare("SELECT 
                            COUNT(DISTINCT v.id) as total_visitas,
                            COUNT(DISTINCT v.mascota_id) as mascotas_atendidas,
                            COALESCE(SUM(f.total), 0) as ingresos_totales,
                            COALESCE(AVG(f.total), 0) as ticket_promedio
                         FROM visitas v
                         LEFT JOIN facturas f ON v.id = f.visita_id AND f.estado = 'pagada'
                         WHERE DATE(v.fecha_visita) BETWEEN :fecha_desde AND :fecha_hasta");
    $stmt->execute(['fecha_desde' => $fecha_desde, 'fecha_hasta' => $fecha_hasta]);
    $resumen = $stmt->fetch();
    
} catch (PDOException $e) {
    die("Error en la base de datos: " . $e->getMessage());
}

include '../includes/header.php';
?>

<div class="row mb-4">
    <div class="col-md-6">
        <h1><i class="bi bi-file-earmark-bar-graph"></i> Reportes y Estadísticas</h1>
    </div>
    <div class="col-md-6 text-end">
        <button type="button" class="btn btn-success" onclick="exportarReporte()">
            <i class="bi bi-file-earmark-excel"></i> Exportar
        </button>
        <button type="button" class="btn btn-primary" onclick="imprimirReporte()">
            <i class="bi bi-printer"></i> Imprimir
        </button>
    </div>
</div>

<!-- Filtros -->
<div class="card mb-4">
    <div class="card-body">
        <form method="GET" action="" class="row g-3">
            <div class="col-md-3">
                <label class="form-label">Período Rápido</label>
                <select class="form-select" name="periodo" id="periodo" onchange="cambiarPeriodo()">
                    <option value="hoy" <?php echo $periodo === 'hoy' ? 'selected' : ''; ?>>Hoy</option>
                    <option value="semana" <?php echo $periodo === 'semana' ? 'selected' : ''; ?>>Esta Semana</option>
                    <option value="mes" <?php echo $periodo === 'mes' ? 'selected' : ''; ?>>Este Mes</option>
                    <option value="ano" <?php echo $periodo === 'ano' ? 'selected' : ''; ?>>Este Año</option>
                    <option value="personalizado" <?php echo $periodo === 'personalizado' ? 'selected' : ''; ?>>Personalizado</option>
                </select>
            </div>
            <div class="col-md-3">
                <label class="form-label">Fecha Desde</label>
                <input type="date" class="form-control" name="fecha_desde" id="fecha_desde" value="<?php echo $fecha_desde; ?>">
            </div>
            <div class="col-md-3">
                <label class="form-label">Fecha Hasta</label>
                <input type="date" class="form-control" name="fecha_hasta" id="fecha_hasta" value="<?php echo $fecha_hasta; ?>">
            </div>
            <div class="col-md-3">
                <label class="form-label">&nbsp;</label>
                <button type="submit" class="btn btn-primary w-100">
                    <i class="bi bi-search"></i> Filtrar
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Resumen General -->
<div class="row mb-4">
    <div class="col-md-3">
        <div class="card bg-primary text-white">
            <div class="card-body">
                <h6 class="card-subtitle mb-2">Total Visitas</h6>
                <h2 class="card-title mb-0"><?php echo formatear_numero($resumen['total_visitas']); ?></h2>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card bg-info text-white">
            <div class="card-body">
                <h6 class="card-subtitle mb-2">Mascotas Atendidas</h6>
                <h2 class="card-title mb-0"><?php echo formatear_numero($resumen['mascotas_atendidas']); ?></h2>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card bg-success text-white">
            <div class="card-body">
                <h6 class="card-subtitle mb-2">Ingresos Totales</h6>
                <h2 class="card-title mb-0"><?php echo formatear_moneda($resumen['ingresos_totales']); ?></h2>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card bg-warning text-white">
            <div class="card-body">
                <h6 class="card-subtitle mb-2">Ticket Promedio</h6>
                <h2 class="card-title mb-0"><?php echo formatear_moneda($resumen['ticket_promedio']); ?></h2>
            </div>
        </div>
    </div>
</div>

<div class="row">
    <!-- Ingresos Diarios -->
    <div class="col-md-6 mb-4">
        <div class="card">
            <div class="card-header bg-primary text-white">
                <h5 class="mb-0">Ingresos Diarios</h5>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-sm">
                        <thead>
                            <tr>
                                <th>Fecha</th>
                                <th>Facturas</th>
                                <th>Total</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($ingresos_diarios as $ingreso): ?>
                                <tr>
                                    <td><?php echo formatear_fecha($ingreso['fecha']); ?></td>
                                    <td><?php echo formatear_numero($ingreso['cantidad_facturas']); ?></td>
                                    <td><strong><?php echo formatear_moneda($ingreso['total_ingresos']); ?></strong></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Tratamientos Más Realizados -->
    <div class="col-md-6 mb-4">
        <div class="card">
            <div class="card-header bg-success text-white">
                <h5 class="mb-0">Tratamientos Más Realizados</h5>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-sm">
                        <thead>
                            <tr>
                                <th>Tratamiento</th>
                                <th>Cantidad</th>
                                <th>Total</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($tratamientos_top as $trat): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($trat['nombre']); ?></td>
                                    <td><?php echo formatear_numero($trat['cantidad']); ?></td>
                                    <td><?php echo formatear_moneda($trat['total_ingresos']); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Medicamentos Más Utilizados -->
    <div class="col-md-6 mb-4">
        <div class="card">
            <div class="card-header bg-info text-white">
                <h5 class="mb-0">Medicamentos Más Utilizados</h5>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-sm">
                        <thead>
                            <tr>
                                <th>Medicamento</th>
                                <th>Cantidad</th>
                                <th>Total</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($medicamentos_top as $med): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($med['nombre']); ?></td>
                                    <td><?php echo formatear_numero($med['cantidad_total']); ?></td>
                                    <td><?php echo formatear_moneda($med['total_ingresos']); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Estadísticas por Profesional -->
    <div class="col-md-6 mb-4">
        <div class="card">
            <div class="card-header bg-warning text-white">
                <h5 class="mb-0">Estadísticas por Profesional</h5>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-sm">
                        <thead>
                            <tr>
                                <th>Profesional</th>
                                <th>Visitas</th>
                                <th>Ingresos</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($profesionales_stats as $prof): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($prof['nombre_completo']); ?></td>
                                    <td><?php echo formatear_numero($prof['cantidad_visitas']); ?></td>
                                    <td><?php echo formatear_moneda($prof['total_ingresos']); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
function cambiarPeriodo() {
    const periodo = document.getElementById('periodo').value;
    const hoy = new Date();
    let desde, hasta;
    
    switch(periodo) {
        case 'hoy':
            desde = hasta = hoy.toISOString().split('T')[0];
            break;
        case 'semana':
            const primerDia = new Date(hoy.setDate(hoy.getDate() - hoy.getDay()));
            desde = primerDia.toISOString().split('T')[0];
            hasta = new Date().toISOString().split('T')[0];
            break;
        case 'mes':
            desde = new Date(hoy.getFullYear(), hoy.getMonth(), 1).toISOString().split('T')[0];
            hasta = new Date().toISOString().split('T')[0];
            break;
        case 'ano':
            desde = new Date(hoy.getFullYear(), 0, 1).toISOString().split('T')[0];
            hasta = new Date().toISOString().split('T')[0];
            break;
        default:
            return;
    }
    
    document.getElementById('fecha_desde').value = desde;
    document.getElementById('fecha_hasta').value = hasta;
}

function exportarReporte() {
    alert('Función de exportación en desarrollo');
}

function imprimirReporte() {
    window.print();
}
</script>

<?php include '../includes/footer.php'; ?>
