<?php
/**
 * Gestión de Promociones
 * Sistema de Administración de Veterinaria
 */

require_once '../config/config.php';
requerir_rol('administrador');

$titulo_pagina = 'Gestión de Promociones';

// Procesar acciones
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $accion = $_POST['accion'] ?? '';
    
    try {
        $db = Database::getInstance()->getConnection();
        
        if ($accion === 'crear') {
            $nombre = limpiar_entrada($_POST['nombre']);
            $descripcion = limpiar_entrada($_POST['descripcion']);
            $tipo_descuento = limpiar_entrada($_POST['tipo_descuento']);
            $valor_descuento = (float)$_POST['valor_descuento'];
            $fecha_inicio = $_POST['fecha_inicio'];
            $fecha_fin = $_POST['fecha_fin'];
            $tratamientos = $_POST['tratamientos'] ?? [];
            
            $stmt = $db->prepare("INSERT INTO promociones (nombre, descripcion, tipo_descuento, valor_descuento, fecha_inicio, fecha_fin) 
                                  VALUES (:nombre, :descripcion, :tipo_descuento, :valor_descuento, :fecha_inicio, :fecha_fin)");
            $stmt->execute([
                'nombre' => $nombre,
                'descripcion' => $descripcion,
                'tipo_descuento' => $tipo_descuento,
                'valor_descuento' => $valor_descuento,
                'fecha_inicio' => $fecha_inicio,
                'fecha_fin' => $fecha_fin
            ]);
            
            $promocion_id = $db->lastInsertId();
            
            // Asociar tratamientos
            if (!empty($tratamientos)) {
                $stmt = $db->prepare("INSERT INTO promocion_tratamientos (promocion_id, tratamiento_id) VALUES (:promocion_id, :tratamiento_id)");
                foreach ($tratamientos as $tratamiento_id) {
                    $stmt->execute(['promocion_id' => $promocion_id, 'tratamiento_id' => $tratamiento_id]);
                }
            }
            
            redirigir_con_mensaje('promociones.php', 'Promoción creada exitosamente', 'success');
            
        } elseif ($accion === 'editar') {
            $id = (int)$_POST['id'];
            $nombre = limpiar_entrada($_POST['nombre']);
            $descripcion = limpiar_entrada($_POST['descripcion']);
            $tipo_descuento = limpiar_entrada($_POST['tipo_descuento']);
            $valor_descuento = (float)$_POST['valor_descuento'];
            $fecha_inicio = $_POST['fecha_inicio'];
            $fecha_fin = $_POST['fecha_fin'];
            $activo = isset($_POST['activo']) ? 1 : 0;
            $tratamientos = $_POST['tratamientos'] ?? [];
            
            $stmt = $db->prepare("UPDATE promociones 
                                  SET nombre = :nombre, descripcion = :descripcion, 
                                      tipo_descuento = :tipo_descuento, valor_descuento = :valor_descuento,
                                      fecha_inicio = :fecha_inicio, fecha_fin = :fecha_fin, activo = :activo
                                  WHERE id = :id");
            $stmt->execute([
                'nombre' => $nombre,
                'descripcion' => $descripcion,
                'tipo_descuento' => $tipo_descuento,
                'valor_descuento' => $valor_descuento,
                'fecha_inicio' => $fecha_inicio,
                'fecha_fin' => $fecha_fin,
                'activo' => $activo,
                'id' => $id
            ]);
            
            // Actualizar tratamientos asociados
            $stmt = $db->prepare("DELETE FROM promocion_tratamientos WHERE promocion_id = :promocion_id");
            $stmt->execute(['promocion_id' => $id]);
            
            if (!empty($tratamientos)) {
                $stmt = $db->prepare("INSERT INTO promocion_tratamientos (promocion_id, tratamiento_id) VALUES (:promocion_id, :tratamiento_id)");
                foreach ($tratamientos as $tratamiento_id) {
                    $stmt->execute(['promocion_id' => $id, 'tratamiento_id' => $tratamiento_id]);
                }
            }
            
            redirigir_con_mensaje('promociones.php', 'Promoción actualizada exitosamente', 'success');
            
        } elseif ($accion === 'eliminar') {
            $id = (int)$_POST['id'];
            $stmt = $db->prepare("DELETE FROM promociones WHERE id = :id");
            $stmt->execute(['id' => $id]);
            
            redirigir_con_mensaje('promociones.php', 'Promoción eliminada exitosamente', 'success');
        }
        
    } catch (PDOException $e) {
        redirigir_con_mensaje('promociones.php', 'Error: ' . $e->getMessage(), 'danger');
    }
}

// Obtener promociones y tratamientos
try {
    $db = Database::getInstance()->getConnection();
    $stmt = $db->query("SELECT * FROM promociones ORDER BY fecha_inicio DESC");
    $promociones = $stmt->fetchAll();
    
    $stmt = $db->query("SELECT * FROM tratamientos WHERE activo = 1 ORDER BY nombre");
    $tratamientos = $stmt->fetchAll();
    
    // Obtener tratamientos asociados a cada promoción
    $promocion_tratamientos = [];
    foreach ($promociones as $promo) {
        $stmt = $db->prepare("SELECT tratamiento_id FROM promocion_tratamientos WHERE promocion_id = :promocion_id");
        $stmt->execute(['promocion_id' => $promo['id']]);
        $promocion_tratamientos[$promo['id']] = $stmt->fetchAll(PDO::FETCH_COLUMN);
    }
    
} catch (PDOException $e) {
    die("Error en la base de datos: " . $e->getMessage());
}

include '../includes/header.php';
?>

<div class="row mb-4">
    <div class="col-md-6">
        <h1><i class="bi bi-tag"></i> Gestión de Promociones</h1>
    </div>
    <div class="col-md-6 text-end">
        <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#modalNuevaPromocion">
            <i class="bi bi-plus-circle"></i> Nueva Promoción
        </button>
    </div>
</div>

<div class="row">
    <?php foreach ($promociones as $promo): ?>
        <?php
        $hoy = date('Y-m-d');
        $vigente = ($promo['fecha_inicio'] <= $hoy && $promo['fecha_fin'] >= $hoy && $promo['activo']);
        $futura = ($promo['fecha_inicio'] > $hoy);
        $vencida = ($promo['fecha_fin'] < $hoy);
        ?>
        <div class="col-md-6 mb-3">
            <div class="card <?php echo $vigente ? 'border-success' : ($vencida ? 'border-secondary' : ''); ?>">
                <div class="card-header <?php echo $vigente ? 'bg-success text-white' : ($vencida ? 'bg-secondary text-white' : 'bg-info text-white'); ?>">
                    <div class="d-flex justify-content-between align-items-center">
                        <h5 class="mb-0"><?php echo htmlspecialchars($promo['nombre']); ?></h5>
                        <div>
                            <?php if ($vigente): ?>
                                <span class="badge bg-light text-success">Vigente</span>
                            <?php elseif ($futura): ?>
                                <span class="badge bg-light text-info">Próxima</span>
                            <?php elseif ($vencida): ?>
                                <span class="badge bg-light text-secondary">Vencida</span>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
                <div class="card-body">
                    <p class="card-text"><?php echo htmlspecialchars($promo['descripcion']); ?></p>
                    
                    <div class="mb-2">
                        <strong>Descuento:</strong>
                        <?php if ($promo['tipo_descuento'] === 'porcentaje'): ?>
                            <?php echo formatear_numero($promo['valor_descuento']); ?>%
                        <?php else: ?>
                            <?php echo formatear_moneda($promo['valor_descuento']); ?>
                        <?php endif; ?>
                    </div>
                    
                    <div class="mb-2">
                        <strong>Vigencia:</strong> 
                        <?php echo formatear_fecha($promo['fecha_inicio']); ?> - <?php echo formatear_fecha($promo['fecha_fin']); ?>
                    </div>
                    
                    <?php if (!empty($promocion_tratamientos[$promo['id']])): ?>
                        <div class="mb-2">
                            <strong>Tratamientos aplicables:</strong>
                            <div class="mt-1">
                                <?php
                                $trat_nombres = [];
                                foreach ($tratamientos as $trat) {
                                    if (in_array($trat['id'], $promocion_tratamientos[$promo['id']])) {
                                        $trat_nombres[] = $trat['nombre'];
                                    }
                                }
                                echo '<small>' . implode(', ', $trat_nombres) . '</small>';
                                ?>
                            </div>
                        </div>
                    <?php endif; ?>
                    
                    <div class="mt-3">
                        <button type="button" class="btn btn-sm btn-warning" 
                                onclick='editarPromocion(<?php echo json_encode($promo); ?>, <?php echo json_encode($promocion_tratamientos[$promo['id']]); ?>)'>
                            <i class="bi bi-pencil"></i> Editar
                        </button>
                        <button type="button" class="btn btn-sm btn-danger" 
                                onclick="eliminarPromocion(<?php echo $promo['id']; ?>, '<?php echo htmlspecialchars($promo['nombre']); ?>')">
                            <i class="bi bi-trash"></i> Eliminar
                        </button>
                    </div>
                </div>
            </div>
        </div>
    <?php endforeach; ?>
</div>

<!-- Modal Nueva Promoción -->
<div class="modal fade" id="modalNuevaPromocion" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <form method="POST" action="">
                <input type="hidden" name="accion" value="crear">
                <div class="modal-header">
                    <h5 class="modal-title">Nueva Promoción</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label">Nombre *</label>
                        <input type="text" class="form-control" name="nombre" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Descripción</label>
                        <textarea class="form-control" name="descripcion" rows="2"></textarea>
                    </div>
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Tipo de Descuento *</label>
                            <select class="form-select" name="tipo_descuento" id="tipo_descuento" required>
                                <option value="porcentaje">Porcentaje (%)</option>
                                <option value="monto_fijo">Monto Fijo (<?php echo MONEDA; ?>)</option>
                            </select>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Valor del Descuento *</label>
                            <input type="number" class="form-control" name="valor_descuento" step="1" required>
                        </div>
                    </div>
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Fecha Inicio *</label>
                            <input type="date" class="form-control" name="fecha_inicio" required>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Fecha Fin *</label>
                            <input type="date" class="form-control" name="fecha_fin" required>
                        </div>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Tratamientos Aplicables</label>
                        <div class="border p-3" style="max-height: 200px; overflow-y: auto;">
                            <?php foreach ($tratamientos as $trat): ?>
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" name="tratamientos[]" 
                                           value="<?php echo $trat['id']; ?>" id="trat_<?php echo $trat['id']; ?>">
                                    <label class="form-check-label" for="trat_<?php echo $trat['id']; ?>">
                                        <?php echo htmlspecialchars($trat['nombre']); ?>
                                    </label>
                                </div>
                            <?php endforeach; ?>
                        </div>
                        <small class="text-muted">Si no selecciona ninguno, la promoción aplicará a todos los tratamientos</small>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-primary">Guardar</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Modal Editar Promoción -->
<div class="modal fade" id="modalEditarPromocion" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <form method="POST" action="">
                <input type="hidden" name="accion" value="editar">
                <input type="hidden" name="id" id="edit_id">
                <div class="modal-header">
                    <h5 class="modal-title">Editar Promoción</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label">Nombre *</label>
                        <input type="text" class="form-control" name="nombre" id="edit_nombre" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Descripción</label>
                        <textarea class="form-control" name="descripcion" id="edit_descripcion" rows="2"></textarea>
                    </div>
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Tipo de Descuento *</label>
                            <select class="form-select" name="tipo_descuento" id="edit_tipo_descuento" required>
                                <option value="porcentaje">Porcentaje (%)</option>
                                <option value="monto_fijo">Monto Fijo (<?php echo MONEDA; ?>)</option>
                            </select>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Valor del Descuento *</label>
                            <input type="number" class="form-control" name="valor_descuento" id="edit_valor_descuento" step="1" required>
                        </div>
                    </div>
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Fecha Inicio *</label>
                            <input type="date" class="form-control" name="fecha_inicio" id="edit_fecha_inicio" required>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Fecha Fin *</label>
                            <input type="date" class="form-control" name="fecha_fin" id="edit_fecha_fin" required>
                        </div>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Tratamientos Aplicables</label>
                        <div class="border p-3" style="max-height: 200px; overflow-y: auto;" id="edit_tratamientos_container">
                            <?php foreach ($tratamientos as $trat): ?>
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" name="tratamientos[]" 
                                           value="<?php echo $trat['id']; ?>" id="edit_trat_<?php echo $trat['id']; ?>">
                                    <label class="form-check-label" for="edit_trat_<?php echo $trat['id']; ?>">
                                        <?php echo htmlspecialchars($trat['nombre']); ?>
                                    </label>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                    <div class="mb-3">
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" name="activo" id="edit_activo">
                            <label class="form-check-label" for="edit_activo">Activo</label>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-primary">Actualizar</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Formulario oculto para eliminar -->
<form method="POST" action="" id="formEliminar" style="display: none;">
    <input type="hidden" name="accion" value="eliminar">
    <input type="hidden" name="id" id="delete_id">
</form>

<script>
function editarPromocion(promo, tratamientosIds) {
    document.getElementById('edit_id').value = promo.id;
    document.getElementById('edit_nombre').value = promo.nombre;
    document.getElementById('edit_descripcion').value = promo.descripcion || '';
    document.getElementById('edit_tipo_descuento').value = promo.tipo_descuento;
    document.getElementById('edit_valor_descuento').value = promo.valor_descuento;
    document.getElementById('edit_fecha_inicio').value = promo.fecha_inicio;
    document.getElementById('edit_fecha_fin').value = promo.fecha_fin;
    document.getElementById('edit_activo').checked = promo.activo == 1;
    
    // Marcar tratamientos seleccionados
    document.querySelectorAll('#edit_tratamientos_container input[type="checkbox"]').forEach(function(checkbox) {
        checkbox.checked = tratamientosIds.includes(parseInt(checkbox.value));
    });
    
    new bootstrap.Modal(document.getElementById('modalEditarPromocion')).show();
}

function eliminarPromocion(id, nombre) {
    if (confirm('¿Está seguro de eliminar la promoción "' + nombre + '"?')) {
        document.getElementById('delete_id').value = id;
        document.getElementById('formEliminar').submit();
    }
}
</script>

<?php include '../includes/footer.php'; ?>
